<?php
/**
 * IPT FSQM Export API Class
 *
 * @package eForm - WordPress Form Builder
 * @subpackage Exporter Static APIs
 * @author Swashata Ghosh <swashata@intechgrity.com>
 */

class IPT_FSQM_EXP_Export_API {

	/**
	 * The User Interface instance for admin area from the mother plugin
	 *
	 * @var IPT_Plugin_UIF_Admin
	 */
	static $ui = null;

	/*==========================================================================
	 * Hooks and Filters Integration
	 *========================================================================*/

	public static function admin_load() {
		// Add the dashboard widget
		add_action( 'toplevel_page_ipt_fsqm_dashboard_page_after', array( __CLASS__, 'ipt_fsqm_dashboard' ), $priority = 10, 1 );
		// Exporter generator
		add_action( 'wp_ajax_ipt_fsqm_exp_export_report', array( __CLASS__, 'ipt_fsqm_exp_export_report_cb' ) );
		// Exporter create and stream file
		add_action( 'wp_ajax_ipt_fsqm_exp_export_report_to_file', array( __CLASS__, 'ipt_fsqm_exp_export_report_to_file_cb' ) );

		// Hook to FSQM Pro Form Deleted
		add_action( 'ipt_fsqm_form_deleted', array( __CLASS__, 'sync_form_delete' ) );
		add_action( 'ipt_fsqm_forms_deleted', array( __CLASS__, 'sync_form_delete' ) );

		// Add our item to the FSQM Shortcode Wizard
		add_action( 'ipt_fsqm_shortcode_wizard', array( __CLASS__, 'persistent_shortcode_wizard_cb' ) );
		add_action( 'wp_ajax_ipt_fsqm_exp_shortcode_insert_persistent_trends', array( __CLASS__, 'persistent_shortcode_generator_cb' ) );

		// Add to new tinyMCE buttons
		add_action( 'ipt_fsqm_tmce_extendor_script', array( __CLASS__, 'persistent_shortcode_script' ) );

		// Add the AJAX for CSV Export
		add_action( 'wp_ajax_ipt_fsqm_exp_raw_csv', array( __CLASS__, 'csv_export_ajax_cb' ) );
		add_action( 'wp_ajax_ipt_fsqm_exp_csv_download', array( __CLASS__, 'csv_download_ajax_cb' ) );
		add_action( 'wp_ajax_ipt_fsqm_exp_xlsx_download', array( __CLASS__, 'xlsx_download_ajax_cb' ) );

		// Add the AJAX for tmce helper
		add_action( 'wp_ajax_ipt_fsqm_exp_export_tmce_details', array( __CLASS__, 'ptrends_tmce_details' ) );
	}

	public static function common_init() {
		global $ipt_fsqm_exp_settings;
		// Persistent Report AJAX Callback
		add_action( 'wp_ajax_ipt_fsqm_exp_report', array( __CLASS__, 'persistent_report_callback_ajax' ) );
		add_action( 'wp_ajax_nopriv_ipt_fsqm_exp_report', array( __CLASS__, 'persistent_report_callback_ajax' ) );

		// Shortcode for Persistent Trends
		add_shortcode( 'ipt_fsqm_ptrends', array( __CLASS__, 'persistent_trends_cb' ) );

		if ( $ipt_fsqm_exp_settings['download_pdf'] == true || is_admin() ) {
			add_filter( 'ipt_fsqm_filter_static_report_print', array( __CLASS__, 'pdf_in_trackback' ), $priority = 10, 2 );
			add_action( 'wp_ajax_ipt_fsqm_exp_download_submission', array( __CLASS__, 'download_pdf_ajax_cb' ) );
			add_action( 'ipt_fsqm_all_data_row_action', array( __CLASS__, 'data_row_action' ), 10, 2 );
			if ( $ipt_fsqm_exp_settings['download_pdf'] == true ) {
				add_filter( 'ipt_fsqm_up_filter_action_button', array( __CLASS__, 'pdf_in_trackback' ), $priority = 10, 2 );
				add_action( 'wp_ajax_nopriv_ipt_fsqm_exp_download_submission', array( __CLASS__, 'download_pdf_ajax_cb' ) );
			}
		}

		// Settings Modifier for PDF attachments
		add_filter( 'ipt_fsqm_filter_default_settings', array( __CLASS__, 'exporter_settings_pdf_downloads' ), 10, 2 );
		add_action( 'ipt_eform_settings_admin', array( __CLASS__, 'exporter_settings_admin_pdf' ), 10, 2 );
		add_action( 'ipt_eform_settings_user', array( __CLASS__, 'exporter_settings_user_pdf' ), 10, 2 );

		// Filter for PDF attachment
		add_filter( 'ipt_fsqm_user_email', array( __CLASS__, 'user_email_attachment_pdf' ), 10, 2 );
		add_filter( 'ipt_fsqm_admin_email', array( __CLASS__, 'admin_email_attachment_pdf' ), 10, 2 );

	}

	/*==========================================================================
	 * Settings Modifier
	 *========================================================================*/
	public static function exporter_settings_pdf_downloads( $settings, $form_id ) {
		$settings['user']['attach_pdf'] = false;
		$settings['admin']['attach_pdf'] = false;
		return $settings;
	}

	public static function exporter_settings_admin_pdf( $op, $form ) {
		?>
		<tr>
			<th><?php $form->ui->generate_label( 'settings[admin][attach_pdf]', __( 'Attach PDF', 'ipt_fsqm' ) ); ?></th>
			<td>
				<?php $form->ui->toggle( 'settings[admin][attach_pdf]', __( 'Yes', 'ipt_fsqm' ), __( 'No', 'ipt_fsqm' ), $op['attach_pdf'] ); ?>
			</td>
			<td>
				<?php $form->ui->help( __( 'Enable if you want to attach PDF with admin notification email.', 'ipt_fsqm' ) ); ?>
			</td>
		</tr>
		<?php
	}

	public static function exporter_settings_user_pdf( $op, $form ) {
		?>
		<tr>
			<th><?php $form->ui->generate_label( 'settings[user][attach_pdf]', __( 'Attach PDF', 'ipt_fsqm' ) ); ?></th>
			<td>
				<?php $form->ui->toggle( 'settings[user][attach_pdf]', __( 'Yes', 'ipt_fsqm' ), __( 'No', 'ipt_fsqm' ), $op['attach_pdf'] ); ?>
			</td>
			<td>
				<?php $form->ui->help( __( 'Enable if you want to attach PDF with user notification email.', 'ipt_fsqm' ) ); ?>
			</td>
		</tr>
		<?php
	}


	/*==========================================================================
	 * Admin Side Row Actions
	 *========================================================================*/
	public static function data_row_action( $rows, $item ) {
		$rows = array_slice( $rows, 0, 3, true ) +
			array( 'download' => '<a href="' . wp_nonce_url( admin_url( 'admin-ajax.php?action=ipt_fsqm_exp_download_submission&id=' . $item['id'] ), 'ipt_fsqm_exp_download_sub_' . $item['id'] ) . '" target="_blank">' . __( 'Download PDF', 'ipt_fsqm' ) . '</a>' ) +
			array_slice( $rows, 3, count( $rows ) - 1, true );
		return $rows;
	}


	/*==========================================================================
	 * CSV Download Function
	 *========================================================================*/
	public static function csv_download_ajax_cb() {
		if ( ! current_user_can( 'manage_feedback' ) ) {
			die( __( 'Cheatin&#8217; uh?' ) );
		}

		$raw_id = isset( $_REQUEST['id'] ) ? (int) $_REQUEST['id'] : 0;

		if ( ! wp_verify_nonce( $_REQUEST['_wpnonce'], 'ipt_fsqm_exp_csv_download_' . $raw_id ) ) {
			die( __( 'Cheatin&#8217; uh?' ) );
		}

		$csv_stream = new IPT_FSQM_EXP_Form_Elements_Export_RAW( $raw_id );
		$csv_stream->stream_csv();
	}

	public static function xlsx_download_ajax_cb() {
		if ( ! current_user_can( 'manage_feedback' ) ) {
			die( __( 'Cheatin&#8217; uh?' ) );
		}

		$raw_id = isset( $_REQUEST['id'] ) ? (int) $_REQUEST['id'] : 0;

		if ( ! wp_verify_nonce( $_REQUEST['_wpnonce'], 'ipt_fsqm_exp_xlsx_download_' . $raw_id ) ) {
			die( __( 'Cheatin&#8217; uh?' ) );
		}

		$csv_stream = new IPT_FSQM_EXP_Form_Elements_Export_RAW( $raw_id );
		$csv_stream->stream_xlsx();
	}


	/*==========================================================================
	 * CSV Export Functions
	 *========================================================================*/
	public static function csv_export_ajax_cb() {
		//Get the variables
		$mcq = isset( $_POST['mcq'] ) && is_array( $_POST['mcq'] ) ? $_POST['mcq'] : array();
		$freetype = isset( $_POST['freetype'] ) && is_array( $_POST['freetype'] ) ? $_POST['freetype'] : array();
		$pinfo = isset( $_POST['pinfo'] ) && is_array( $_POST['pinfo'] ) ? $_POST['pinfo'] : array();
		$settings = isset( $_POST['settings'] ) && is_array( $_POST['settings'] ) ? $_POST['settings'] : array();
		$raw_id = isset( $_POST['raw_id'] ) && !empty( $_POST['raw_id'] ) ? (int) $_POST['raw_id'] : null;
		$raw_nonce = isset( $_POST['raw_nonce'] ) && is_string( $_POST['raw_nonce'] ) ? $_POST['raw_nonce'] : '';
		$doing = isset( $_POST['doing'] ) ? (int) $_POST['doing'] : 0;

		// Check for authenticity
		if ( ! current_user_can( 'manage_feedback' ) ) {
			die( __( 'Cheatin&#8217; uh?' ) );
		}

		// Check for nonce
		if ( ! wp_verify_nonce( $_POST['wpnonce'], 'ipt_fsqm_exp_raw_ajax_' . $settings['form_id'] ) ) {
			die( __( 'Cheatin&#8217; uh?' ) );
		}

		if ( $raw_id !== null ) {
			if ( ! wp_verify_nonce( $raw_nonce, 'ipt_fsqm_exp_raw_update_ajax_' . $raw_id . $doing ) ) {
				die( __( 'Cheatin&#8217; uh?' ) );
			}
		}

		// All done, now prepare the stuff
		$load = $settings['load'];
		$start_date = '';
		$end_date = '';
		if ( $settings['custom_date'] == 'true' ) {
			$start_date = $settings['custom_date_start'];
			$end_date = $settings['custom_date_end'];
		}
		$csv_gen = new IPT_FSQM_EXP_Form_Elements_Export_RAW( $raw_id, $settings['form_id'], false, $start_date, $end_date, $mcq, $freetype, $pinfo );

		if ( $settings['custom_csv'] == 'true' ) {
			$csv_gen->set_delimiter( stripslashes( $settings['custom_csv_delimiter'] ) );
			$csv_gen->set_enclosure( stripslashes( $settings['custom_csv_enclosure'] ) );
			$csv_gen->set_option_delimiter( stripslashes( $settings['custom_csv_option_delimiter'] ) );
			$csv_gen->set_range_delimiter( stripslashes( $settings['custom_csv_range_delimiter'] ) );
			$csv_gen->set_row_delimiter( stripslashes( $settings['custom_csv_row_delimiter'] ) );
			$csv_gen->set_entry_delimiter( stripslashes( $settings['custom_csv_entry_delimiter'] ) );
		}

		$return = array(
			'type' => 'success',
			'done' => 0,
			'html' => '',
			'raw_id' => $csv_gen->get_raw_id(),
			'raw_nonce' => wp_create_nonce( 'ipt_fsqm_exp_raw_update_ajax_' . $csv_gen->get_raw_id() . ( $doing + 1 ) ),
			'download_url' => admin_url( 'admin-ajax.php?action=ipt_fsqm_exp_csv_download&id=' . $csv_gen->get_raw_id() . '&_wpnonce=' . wp_create_nonce( 'ipt_fsqm_exp_csv_download_' . $csv_gen->get_raw_id() ) ),
			'xlsx_url' => admin_url( 'admin-ajax.php?action=ipt_fsqm_exp_xlsx_download&id=' . $csv_gen->get_raw_id() . '&_wpnonce=' . wp_create_nonce( 'ipt_fsqm_exp_xlsx_download_' . $csv_gen->get_raw_id() ) ),
		);

		// Check for errors first
		$errors = $csv_gen->get_errors();
		if ( empty( $errors ) ) {
			$return['done'] = $csv_gen->make_csv( $doing, $load );
		} else {
			self::$ui = IPT_Plugin_UIF_Admin::instance( 'ipt_fsqm' );
			$return['type'] = 'error';
			$return['html'] = self::$ui->msg_error( implode( '<br />', $csv_gen->get_errors() ), false );
		}

		@header( 'Content-Type: application/json; charset=' . get_option( 'blog_charset' ) );
		echo json_encode( (object) $return );
		die();
	}

	/*==========================================================================
	 * Trackback / Full Preview Print Filter
	 *========================================================================*/

	public static function pdf_in_trackback( $buttons, $form ) {
		if ( !is_object( $form ) || ( ! $form instanceof IPT_FSQM_Form_Elements_Front && ! $form instanceof IPT_FSQM_Form_Elements_Data ) ) {
			return $buttons;
		}

		$size = 'auto';
		$text = __( 'Download', 'ipt_fsqm_exp' );

		if ( $form instanceof IPT_FSQM_Form_Elements_Front ) {
			$size = 'medium';
			$text = __( 'Download PDF', 'ipt_fsqm_exp' );
		}

		$buttons[] = array(
			$text,
			'ipt_fsqm_report_download_' . $form->form_id . '_' . $form->data_id,
			$size,
			'secondary',
			'normal',
			array( 'ipt_fsqm_exp_download_submission' ),
			'button',
			array(),
			array('onclick' => 'javascript:window.location.href="' . wp_nonce_url( admin_url( 'admin-ajax.php?action=ipt_fsqm_exp_download_submission&id=' . $form->data_id ), 'ipt_fsqm_exp_download_sub_' . $form->data_id ) . '"'),
			'',
			'file-pdf',
			'before',
		);

		return $buttons;
	}

	public static function get_pdf_html( $data_id, $custom = '', $type = 'user' ) {
		global $ipt_fsqm_exp_settings;
		$data = new IPT_FSQM_Form_Elements_Data( $data_id );

		if ( $data->form_id == null ) {
			return __( 'Error', 'ipt_fsqm_exp' );
		}
		ob_start();
		?>
<!DOCTYPE html PUBLIC "-//W3C//DTD HTML 4.01//EN" "http://www.w3.org/TR/html4/strict.dtd">
<html>
<head>
	<title><?php echo $data->name; ?></title>
	<style type="text/css">
		html {
			margin: 20px;
			font-size: 11pt;
		}
		body {
			margin: 0;
			padding: 0;
		}
		table {
			text-align: left;
			vertical-align: middle;
			width: 100%;
			margin: 10px;
			border-collapse: collapse;
			page-break-before: always;
			overflow: wrap;
		}
		table td,
		table th {
			padding: 10px;
			text-align: left;
			border-collapse: collapse;
		}
		table thead tr th,
		table tfoot tr th,
		table tr.head th {
			color: #fff;
		}
		table td.data {
			padding: 0;
		}
		table th {
			font-weight: bold;
		}
		.data table,
		.matrix table,
		.matrix table tbody,
		.matrix table tr,
		.matrix table thead,
		.matrix table tfoot {
			height: 100%;
			margin: 0;
			width: 100%;
		}
		td.icons {
			width: 1cm;
			padding: 5px;
			overflow: wrap;
		}
		td.icons img {
			display: inline;
			border: 0 none;
		}
		.description {
			font-size: 10pt;
			text-transform: lowercase;
			font-style: italic;
			font-weight: normal;
		}
		<?php echo $ipt_fsqm_exp_settings['css']; ?>
	</style>
</head>
<body>
	<?php echo wpautop( $custom ); ?>
	<?php echo $ipt_fsqm_exp_settings['html_header']; ?>
	<?php $data->show_quick_preview( true, true, true, ( ( 'admin' == $type ) ? true : false ) ); ?>
	<?php echo $ipt_fsqm_exp_settings['html_footer']; ?>
</body>
</html>
		<?php
		return ob_get_clean();
	}

	public static function download_pdf_ajax_cb() {
		if ( ! isset( $_REQUEST['id'] ) ) {
			die( __( 'Cheatin&#8217; uh?' ) );
		}
		$data_id = (int) $_REQUEST['id'];

		if ( ! wp_verify_nonce( $_REQUEST['_wpnonce'], 'ipt_fsqm_exp_download_sub_' . $data_id ) ) {
			die( __( 'Cheatin&#8217; uh?' ) );
		}

		global $ipt_fsqm_exp_settings;

		require_once IPT_FSQM_EXP_Loader::$abs_path . '/lib/vendor/autoload.php';

		$type = 'user';
		if ( is_user_logged_in() && current_user_can( 'manage_feedback' ) ) {
			$type = 'admin';
		}

		$html = self::get_pdf_html( $data_id, '', $type );
		$pdf = new \Mpdf\Mpdf( [ 'mode' => 'utf-8', 'format' => 'A4' ] );
		$pdf->shrink_tables_to_fit = 1;
		if ( $ipt_fsqm_exp_settings['landscape'] == true ) {
			$pdf->AddPage('L');
		}
		$pdf->WriteHTML( $html, 0 );
		$filename = sanitize_file_name( $ipt_fsqm_exp_settings['pdf_prefix'] . $data_id . '.pdf' );
		$pdf->Output( $filename, 'D' );
		die();
	}

	public static function email_attachment_pdf( $type, $email, $data ) {
		global $ipt_fsqm_exp_settings;

		if ( 'user' != $type && 'admin' != $type ) {
			$type = 'user';
		}

		if ( true != $data->settings[ $type ]['attach_pdf'] ) {
			return $email;
		}

		// Check status
		if ( 'admin' != $type && true == $data->get_submission_lock_status() ) {
			return $email;
		}

		$filename = trailingslashit( sys_get_temp_dir() ) . $ipt_fsqm_exp_settings['pdf_prefix'] . date( 'Y-m-d', current_time( 'timestamp' ) ) . '-' . $type . '-' . $data->data_id . '.pdf';
		// Delete the file if it exists
		// Resolves #39
		if ( file_exists( $filename ) ) {
			@unlink( $filename );
		}
		if ( ! file_exists( $filename ) ) {
			// First create the file
			require_once IPT_FSQM_EXP_Loader::$abs_path . '/lib/vendor/autoload.php';

			$html = self::get_pdf_html( $data->data_id, '', $type );
			$pdf = new \Mpdf\Mpdf( [ 'mode' => 'utf-8', 'format' => 'A4' ] );
			$pdf->shrink_tables_to_fit = 1;
			if ( $ipt_fsqm_exp_settings['landscape'] == true ) {
				$pdf->AddPage('L');
			}

			$pdf->WriteHTML( $html, 0 );

			$pdf->Output( $filename, 'F' );
		}

		foreach ( $email as $key => $val ) {
			$email[$key]['attachment'][] = $filename;
		}

		return $email;
	}

	public static function user_email_attachment_pdf( $email, $data ) {
		return self::email_attachment_pdf( 'user', $email, $data );
	}

	public static function admin_email_attachment_pdf( $email, $data ) {
		return self::email_attachment_pdf( 'admin', $email, $data );
	}


	/*==========================================================================
	 * Shortcode Wizard Callbacks
	 *========================================================================*/
	public static function persistent_shortcode_generator_cb() {
		$exports = self::get_exports();
		?>
<table class="form-table" id="ipt_fsqm_exp_shortcode_table">
	<tbody>
		<tr>
			<th><label for="ipt_fsqm_exp_shortcode_export"><?php _e( 'Select the Report', 'ipt_fsqm_exp' ); ?></label></th>
			<td>
				<select id="ipt_fsqm_exp_shortcode_export">
					<?php if ( ! empty( $exports ) ) : ?>
					<?php foreach ( $exports as $export ) : ?>
					<option value="<?php echo $export->id; ?>"><?php echo $export->name . ' [' . date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), strtotime( $export->created ) ) . ']'; ?></option>
					<?php endforeach; ?>
					<?php else : ?>
					<option value="0"><?php _e( 'You have not generated any reports yet.', 'ipt_fsqm_exp' ); ?></option>
					<?php endif; ?>
				</select>
			</td>
		</tr>
	</tbody>
	<tfoot>
		<tr>
			<td colspan="2">
				<p class="submit">
					<input type="button" class="button-primary" value="<?php _e( 'Insert', 'ipt_fsqm_exp' ); ?>" id="ipt_fsqm_exp_shortcode_export_insert_button" />
					<a id="ipt_fsqm_shortcode_wizard_back" title="<?php _e( 'Insert Shortcodes for WP Feedback, Survey & Quiz Manager - Pro', 'ipt_fsqm' ); ?>" href="admin-ajax.php?action=ipt_fsqm_shortcode_insert" class="button-secondary"><?php _e( 'Back', 'ipt_fsqm' ); ?></a>
				</p>
			</td>
		</tr>
	</tfoot>
</table>

<script type="text/javascript">
jQuery(document).ready(function($) {
	$('#ipt_fsqm_exp_shortcode_export_insert_button').on('click', function(e) {
		e.preventDefault();
		var val = $('#ipt_fsqm_exp_shortcode_export').val(),
		shortcode = '<br />[ipt_fsqm_ptrends id="' + val + '"]<br />';
		if ( val == 0 || val == '' ) {
			shortcode = '';
		}
		$(document).trigger('ipt_fsqm_shortcode_insert', [function(ed) {
			ed.execCommand('mceInsertContent', 0, shortcode);
		}]);
	});
});
</script>
		<?php
		die();
	}
	public static function persistent_shortcode_wizard_cb() {
		?>
<tr>
	<td>
		<a title="<?php _e( 'Exporter for eForm - Insert Persistent Trends', 'ipt_fsqm_exp' ); ?>" class="button-secondary" href="admin-ajax.php?action=ipt_fsqm_exp_shortcode_insert_persistent_trends"><?php _e( 'Persistent Trends', 'ipt_fsqm_exp' ); ?></a>
	</td>
	<td>
		<span class="description">
			<?php _e( 'Recommended method to show trends, where instead of querying your server for the live data, one of your generated reports will be used. This saves a lot of server bandwidth and load. But please note that the report will only be as updated as your export.', 'ipt_fsqm_exp' ); ?>
		</span>
	</td>
</tr>
		<?php
	}

	public static function persistent_shortcode_script() {
		wp_enqueue_script( 'ipt_fsqm_exp_tmce_fsqm', plugins_url( '/static/admin/js/fsqm-exp-tinymce-plugin.min.js', IPT_FSQM_EXP_Loader::$abs_file ), array( 'jquery' ), IPT_FSQM_EXP_Loader::$version );
		wp_localize_script( 'ipt_fsqm_exp_tmce_fsqm', 'iptFSQMEXPtmce', array(
			'exports' => (array) self::get_exports(),
			'l10n' => array(
				'ps' => __( 'Persistent Trends', 'ipt_fsqm_exp' ),
				'pstt' => __( 'Recommended method to show trends, where instead of querying your server for the live data, one of your generated reports will be used. This saves a lot of server bandwidth and load. But please note that the report will only be as updated as your export.', 'ipt_fsqm_exp' ),
				'pid' => __( 'Select Report', 'ipt_fsqm' ),
			),
			'tmceDetailsNonce' => wp_create_nonce( 'ipt_fsqm_exp_details' ),
		) );
	}

	public static function ptrends_tmce_details() {
		// Get the exp_id
		$exp_id = (int) @$_REQUEST['exp_id'];

		// Verify the nonce
		if ( ! wp_verify_nonce( @$_REQUEST['wpnonce'], 'ipt_fsqm_exp_details' ) ) {
			die( __( 'Cheatin&#8217; uh?' ) );
		}

		// Get the details
		$exp_data = self::get_exp_data( $exp_id );

		// Check the sanity
		if ( false === $exp_data ) {
			wp_die( __( 'Cheatin&#8217; uh?' ) );
		}

		// Initialize the return
		$return = array(
			'mcqs' => array(),
			'freetypes' => array(),
			'pinfos' => array(),
		);

		// Init a form instance
		$form = new IPT_FSQM_Form_Elements_Base( $exp_data->form_id );
		if ( null == $form->form_id ) {
			wp_die( 'Invalid or Deleted Form' );
		}

		// Include mcqs
		if ( in_array( 'mcq', $exp_data->meta['settings']['report'] ) && ! empty( $exp_data->meta['mcqs'] ) ) {
			$rmcqs = array();
			$rmcqs[] = array(
				'text' => __( 'Show All', 'ipt_fsqm' ),
				'value' => 'all',
			);
			foreach ( $exp_data->meta['mcqs'] as $m_key ) {
				if ( isset( $form->mcq[ $m_key ] ) ) {
					$rmcqs[] = array(
						'text' => $form->mcq[ $m_key ]['title'],
						'value' => $m_key,
						'type' => $form->mcq[ $m_key ]['type'],
					);
				}
			}
			$return['mcqs'] = $rmcqs;
		}

		// Include feedbacks
		if ( in_array( 'freetype', $exp_data->meta['settings']['report'] ) && ! empty( $exp_data->meta['freetypes'] ) ) {
			$rfreetypes = array();
			$rfreetypes[] = array(
				'text' => __( 'Show All', 'ipt_fsqm' ),
				'value' => 'all',
			);
			foreach ( $exp_data->meta['freetypes'] as $f_key ) {
				if ( isset( $form->freetype[ $f_key ] ) ) {
					$rfreetypes[] = array(
						'text' => $form->freetype[ $f_key ]['title'],
						'value' => $f_key,
						'type' => $form->freetype[ $f_key ]['type'],
					);
				}
			}
			$return['freetypes'] = $rfreetypes;
		}

		// Include pinfos
		if ( in_array( 'pinfo', $exp_data->meta['settings']['report'] ) && ! empty( $exp_data->meta['pinfos'] ) ) {
			$rpinfos = array();
			$rpinfos[] = array(
				'text' => __( 'Show All', 'ipt_fsqm' ),
				'value' => 'all',
			);
			foreach ( $exp_data->meta['pinfos'] as $p_key ) {
				if ( isset( $form->pinfo[ $p_key ] ) ) {
					$rpinfos[] = array(
						'text' => $form->pinfo[ $p_key ]['title'],
						'value' => $p_key,
						'type' => $form->pinfo[ $p_key ]['type'],
					);
				}
			}
			$return['pinfos'] = $rpinfos;
		}

		// All set
		// Now release
		@header( 'Content-Type: application/json; charset=' . get_option( 'blog_charset' ) );
		echo json_encode( (object) $return );
		die();
	}


	/*==========================================================================
	 * Persistent Reports as Trends
	 *========================================================================*/

	public static function persistent_trends_cb( $atts, $content = null ) {
		global $wpdb, $ipt_fsqm_exp_info;
		// Some backward compatibility before hand
		if ( ! isset( $atts['mcq_ids'] ) ) {
			$atts['mcq_ids'] = 'all';
		}

		// Now parse the shortcode atts
		$atts = shortcode_atts( array(
			'id' => 0,
			'title' => __( 'Trends', 'ipt_fsqm_exp' ),
			'mcq_ids' => '',
			'mcq_config' => '',
			'freetype_ids' => '',
			'pinfo_ids' => '',
			'pinfo_config' => '',
			'data' => '',
			'appearance' => '',
		), $atts );

		// We do our magic now
		$exp_id = $atts['id'];
		// Get the exp data
		$exp_data = self::get_exp_data( $exp_id );
		// If it is not present, then simply exit
		if ( false === $exp_data ) {
			return __( 'Invalid Export ID Provided!', 'ipt_fsqm_exp' );
		}
		// Get the form_id
		$form_id = $exp_data->form_id;

		// Create an instance of front
		$front = new IPT_FSQM_Form_Elements_Front( null, $form_id );

		// Init the parameters
		$persistent_report_param = array( $atts['id'], $front->ui, $atts['title'] );

		// Create the mcq list
		$mcqs = array();
		if ( '' !== $atts['mcq_ids'] && 'all' !== $atts['mcq_ids'] ) {
			$mcqs = wp_parse_id_list( $atts['mcq_ids'] );
		} else if ( 'all' == $atts['mcq_ids'] ) {
			$mcqs = $exp_data->meta['mcqs'];
		}

		// Create the freetype list
		$freetypes = array();
		if ( '' !== $atts['freetype_ids'] && 'all' !== $atts['freetype_ids'] ) {
			$freetypes = wp_parse_id_list( $atts['freetype_ids'] );
		} else if ( 'all' == $atts['freetype_ids'] ) {
			$freetypes = $exp_data->meta['freetypes'];
		}

		// Create the pinfo list
		$pinfos = array();
		if ( '' !== $atts['pinfo_ids'] && 'all' !== $atts['pinfo_ids'] ) {
			$pinfos = wp_parse_id_list( $atts['pinfo_ids'] );
		} else if ( 'all' == $atts['pinfo_ids'] ) {
			$pinfos = $exp_data->meta['pinfos'];
		}

		// Get the default chart types and toggles
		$mcqs_in_report = array();
		foreach ( $mcqs as $m_key ) {
			if ( isset( $front->mcq[ $m_key ] ) ) {
				$mcqs_in_report[ $m_key ] = array(
					'type' => $front->mcq[ $m_key ]['type'],
				);
			}
		}
		$pinfos_in_report = array();
		foreach ( $pinfos as $p_key ) {
			if ( isset( $front->pinfo[ $p_key ] ) ) {
				$pinfos_in_report[ $p_key ] = array(
					'type' => $front->pinfo[ $p_key ]['type'],
				);
			}
		}
		// Set the value in default_chart_n_toggle variable
		$default_chart_n_toggle = IPT_FSQM_Form_Elements_Static::get_default_chart_n_toggle( $mcqs_in_report, $pinfos_in_report );

		// Deserialize the JSONs of chartconfigs
		if ( '' !== $atts['mcq_config'] ) {
			$atts['mcq_config'] = json_decode( $atts['mcq_config'], true );
			if ( is_null( $atts['mcq_config'] ) ) {
				$atts['mcq_config'] = $default_chart_n_toggle['mcq'];
			}
		} else {
			// Backward compatibility
			$atts['mcq_config'] = $default_chart_n_toggle['mcq'];
		}

		if ( '' !== $atts['pinfo_config'] ) {
			$atts['pinfo_config'] = json_decode( $atts['pinfo_config'], true );
			if ( is_null( $atts['pinfo_config'] ) ) {
				$atts['pinfo_config'] = $default_chart_n_toggle['pinfo'];
			}
		} else {
			$atts['pinfo_config'] = $default_chart_n_toggle['pinfo'];
		}

		// Appearance config
		$rappearance = json_decode( $atts['appearance'], true );
		if ( is_null( $rappearance ) ) {
			$rappearance = array();
		}
		$rappearance = wp_parse_args( $rappearance, array(
			'block' => true,
			'heading' => true,
			'description' => true,
			'header' => true,
			'border' => true,
			'material' => false,
			'print' => false,
		) );

		$appearance = array(
			'wrap' => ( true == $rappearance['block'] ) ? true : false,
			'heading' => ( true == $rappearance['heading'] ) ? true : false,
			'description' => ( true == $rappearance['description'] ) ? true : false,
			'theader' => ( true == $rappearance['header'] ) ? true : false,
			'tborder' => ( true == $rappearance['border'] ) ? true : false,
			'material' => ( true == $rappearance['material'] ) ? true : false,
			'print' => ( true == $rappearance['print'] ) ? true : false,
		);

		// Data config
		$data = json_decode( $atts['data'], true );
		if ( is_null( $data ) ) {
			$data = array(
				'data' => true,
				'others' => false,
				'names' => false,
				'date' => false,
			);
		}
		// No sensitive data
		$data['sensitive'] = false;

		// Chart meta config
		$cmeta = array(
			'mcq' => $atts['mcq_config'],
			'pinfo' => $atts['pinfo_config'],
		);

		// Now fill the values
		$persistent_report_param[3] = $mcqs;
		$persistent_report_param[4] = $freetypes;
		$persistent_report_param[5] = $pinfos;
		$persistent_report_param[6] = $appearance;
		$persistent_report_param[7] = $data;
		$persistent_report_param[8] = $cmeta;

		// Prepare to catch the output
		ob_start();
		$front->container( array( array( __CLASS__, 'persistent_report' ), $persistent_report_param ) );
		return ob_get_clean();
	}

	public static function persistent_report( $exp_id, $ui = null, $visualization = '', $mcqs = false, $freetypes = false, $pinfos = false, $appearance = false, $data = false, $cmeta = array() ) {
		$exp_data = self::get_exp_data( $exp_id );

		// Sanitize the UI object
		if ( null === $ui ) {
			if ( is_admin() ) {
				$ui = IPT_Plugin_UIF_Admin::instance( 'ipt_fsqm' );
			} else {
				// Get form class for UI
				$form = new IPT_FSQM_Form_Elements_Front( null, $exp_data->form_id );
				$ui = $form->ui;
			}
		}

		// Check for exp_data sanity
		if ( false === $exp_data ) {
			$ui->msg_error( __( 'Invalid Export ID', 'ipt_fsqm_exp' ) );
			return;
		}

		// Instantiate the util class
		$util = new IPT_FSQM_Form_Elements_Utilities( $exp_data->form_id, $ui );

		// Realize the report types
		// And set the keys
		if ( false === $mcqs ) {
			$mcqs = $exp_data->meta['mcqs'];
		}
		if ( false === $freetypes ) {
			$freetypes = $exp_data->meta['freetypes'];
		}
		if ( false === $pinfos ) {
			$pinfos = $exp_data->meta['pinfos'];
		}
		$report_type = array();
		if ( ! empty( $mcqs ) ) {
			$report_type[] = 'mcq';
		}
		if ( ! empty( $freetypes ) ) {
			$report_type[] = 'freetype';
		}
		if ( ! empty( $pinfos ) ) {
			$report_type[] = 'pinfo';
		}

		// Create the settings
		$settings = array(
			'form_id' => $exp_data->form_id,
			'report' => $report_type,
			'load' => '1',
			'exp_id' => $exp_id,
		);

		// Create the appearance
		if ( false === $appearance ) {
			$appearance = array(
				'wrap' => true,
				'heading' => true,
				'description' => true,
				'theader' => true,
				'tborder' => true,
				'material' => false,
				'print' => true,
			);
		}
		$appearance = wp_parse_args( $appearance, array(
			'wrap' => true,
			'heading' => true,
			'description' => true,
			'theader' => true,
			'tborder' => true,
			'material' => true,
			'print' => true,
		) );

		// Create the data
		if ( false === $data ) {
			$data = array(
				'data' => true,
				'names' => true,
				'date' => true,
				'others' => true,
				'sensitive' => true,
			);
		}
		$data = wp_parse_args( $data, array(
			'data' => true,
			'names' => true,
			'date' => true,
			'others' => true,
			'sensitive' => false,
		) );

		// Create the cmeta
		if ( empty( $cmeta ) ) {
			// Populate the mcqs
			$in_report_mcqs = array();
			foreach ( $mcqs as $m_key ) {
				$in_report_mcqs[ $m_key ] = array(
					'type' => $util->mcq[ "$m_key" ]['type'],
				);
			}
			// Populate the pinfos
			$in_report_pinfos = array();
			foreach ( $pinfos as $p_key ) {
				$in_report_pinfos[ $p_key ] = array(
					'type' => $util->pinfo[ "$p_key" ]['type'],
				);
			}

			$cmeta = IPT_FSQM_Form_Elements_Static::get_default_chart_n_toggle( $in_report_mcqs, $in_report_pinfos );
		}

		$util->report_generate_report( $settings, $mcqs, $freetypes, $pinfos, $data['data'], $data['names'], $data['date'], $data['others'], $data['sensitive'], $appearance, $cmeta, $exp_data->meta['filters'], $visualization, 'ipt_fsqm_exp_report' );
	}

	public static function persistent_report_callback_ajax() {
		global $wpdb, $ipt_fsqm_info, $ipt_fsqm_settings;
		$settings = isset( $_POST['settings'] ) ? $_POST['settings'] : array();
		$doing = isset( $_POST['doing'] ) ? (int) $_POST['doing'] : 0;
		$form_id = isset( $_POST['form_id'] ) ? (int) $_POST['form_id'] : 0;
		$do_data = isset( $_POST['do_data'] ) && 'true' == $_POST['do_data'] ? true : false;
		$do_names = isset( $_POST['do_names'] ) && 'true' == $_POST['do_names'] ? true : false;
		$do_others = isset( $_POST['do_others'] ) && 'true' == $_POST['do_others'] ? true : false;
		$sensitive_data = isset( $_POST['sensitive_data'] ) && 'true' == $_POST['sensitive_data'] ? true : false;
		$do_date = isset( $_POST['do_date'] ) && 'true' == $_POST['do_date'] ? true : false;
		$query_elements = isset( $_POST['query_elements'] ) ? (array) $_POST['query_elements'] : array();
		$query_elements = wp_parse_args( $query_elements, array(
			'mcqs' => array(),
			'freetypes' => array(),
			'pinfos' => array(),
		) );
		$filters = isset( $_POST['filters'] ) ? wp_unslash( $_POST['filters'] ) : array();

		$exp_id = isset( $settings['exp_id'] ) ? $settings['exp_id'] : 0;

		// Check general nonce
		if ( ! wp_verify_nonce( $_POST['wpnonce'], 'ipt_fsqm_report_ajax_' . $form_id ) ) {
			die( __( 'Cheatin&#8217; uh?' ) . ' none' );
		}

		// Verify do_data nonce for showing data alongside mcqs
		if ( $do_data && ! wp_verify_nonce( $_POST['do_data_nonce'], 'ipt_fsqm_report_ajax_do_data_' . $form_id ) ) {
			die( __( 'Cheatin&#8217; uh?' ) . ' do_data' );
		}

		// Verify do_names nonce for showing names to mcq meta and freetype
		if ( $do_names && ! wp_verify_nonce( $_POST['do_names_nonce'], 'ipt_fsqm_report_ajax_do_names_' . $form_id ) ) {
			die( __( 'Cheatin&#8217; uh?' ) . ' do_names' );
		}

		// Verify do_others for populating meta entries
		if ( $do_others && ! wp_verify_nonce( $_POST['do_others_nonce'], 'ipt_fsqm_report_ajax_do_others_' . $form_id ) ) {
			die( __( 'Cheatin&#8217; uh?' ) . ' do_others' );
		}

		// If sensitive data is enabled, then must verify the nonce and also current user should be able to manage_feedback
		if ( $sensitive_data && ( ! wp_verify_nonce( $_POST['sensitive_data_nonce'], 'ipt_fsqm_report_ajax_sensitive_data_' . $form_id ) || ! current_user_can( 'manage_feedback' ) ) ) {
			die( __( 'Cheatin&#8217; uh?' ) . ' sensitive_data' );
		}

		// Verify do_date for populating meta and freetype entries
		if ( $do_date && ! wp_verify_nonce( $_POST['do_date_nonce'], 'ipt_fsqm_report_ajax_do_date_' . $form_id ) ) {
			die( __( 'Cheatin&#8217; uh?' ) . ' do_date' );
		}

		$return = array(
			'type' => 'success',
			'done' => '100',
			'survey' => array(),
			'feedback' => array(),
			'pinfo' => array(),
			'form_id' => $form_id,
			'debug_info' => array(),
		);

		//First test the form_id
		if ( null == $wpdb->get_var( $wpdb->prepare( "SELECT name FROM {$ipt_fsqm_info['form_table']} WHERE id = %d", $form_id ) ) ) {
			die( __( 'Cheatin&#8217; uh?' ) );
		}

		//Get the persistent data
		$exp_data = self::get_exp_data( $exp_id );

		// Test for consistency
		if ( false === $exp_data || $exp_data->form_id != $form_id ) {
			die( __( 'Cheatin&#8217; uh?' ) );
		}

		$util = new IPT_FSQM_Form_Elements_Utilities( $exp_data->form_id );

		// Do the surveys
		if ( is_array( $query_elements['mcqs'] ) && ! empty( $query_elements['mcqs'] ) ) {
			foreach ( $query_elements['mcqs'] as $m_key ) {
				$m_key = (int) $m_key;
				if ( ! isset( $util->mcq[ $m_key ] ) || ! isset( $exp_data->mcqs[ $m_key ] ) ) {
					continue;
				}
				$element = $util->mcq[ $m_key ];

				switch ( $element['type'] ) {
					default :
						$definition = $util->get_element_definition( $element );
						if ( isset( $definition['callback_exporter_persistent_sanitizer'] ) && is_callable( $definition['callback_exporter_persistent_sanitizer'] ) ) {
							$return['survey'][ "$m_key" ] = call_user_func( $definition['callback_exporter_persistent_sanitizer'], $element, $exp_data->mcqs[ $m_key ], $m_key, $do_data, $do_names, $do_others, $sensitive_data, $do_date, $util );
						} else {
							$return['survey'][ "$m_key" ] = $exp_data->mcqs[ "$m_key" ];
						}
						break;
					case 'radio' :
					case 'checkbox' :
					case 'select' :
					case 'thumbselect' :
						if ( ! $do_others && isset( $exp_data->mcqs[ "$m_key" ]['others_data'] ) ) {
							unset( $exp_data->mcqs[ "$m_key" ]['others_data'] );
						} else {
							if ( isset( $exp_data->mcqs[ "$m_key" ]['others_data'] ) ) {
								foreach ( $exp_data->mcqs[ "$m_key" ]['others_data'] as $o_key => $other ) {
									$exp_data->mcqs[ "$m_key" ]['others_data'][ $o_key ] = array(
										'value' => $other['value'],
									);
									if ( $do_names ) {
										$exp_data->mcqs[ "$m_key" ]['others_data'][ $o_key ]['name'] = $other['name'];
									}
									if ( $sensitive_data ) {
										$exp_data->mcqs[ "$m_key" ]['others_data'][ $o_key ]['email'] = ( __( 'anonymous', 'ipt_fsqm_exp' ) !== $other['email'] ? '<a href="mailto:' . $other['email'] . '">' . $other['email'] . '</a>' : $other['email'] );
										$exp_data->mcqs[ "$m_key" ]['others_data'][ $o_key ]['id'] = $other['id'];
									}
									if ( $do_date ) {
										$exp_data->mcqs[ "$m_key" ]['others_data'][ $o_key ]['date'] = isset( $other['date'] ) ? $other['date'] : '';
									}
								}
							}
						}
						$return['survey'][ "$m_key" ] = $exp_data->mcqs[ "$m_key" ];
						break;
					case 'smileyrating' :
					case 'likedislike' :
						if ( ! $do_others && isset( $exp_data->mcqs[ "$m_key" ]['feedback_data'] ) ) {
							unset( $exp_data->mcqs[ "$m_key" ]['feedback_data'] );
						} else {
							if ( isset( $exp_data->mcqs[ "$m_key" ]['feedback_data'] ) ) {
								$setting_to_icon_map = array(
									'frown' => 'angry2',
									'sad' => 'sad2',
									'neutral' => 'neutral2',
									'happy' => 'smiley2',
									'excited' => 'happy2',
								);
								if ( 'likedislike' == $element['type'] ) {
									$setting_to_icon_map = array(
										'like' => 'thumbs-o-up',
										'dislike' => 'thumbs-o-down',
									);
								}
								foreach ( $exp_data->mcqs[ "$m_key" ]['feedback_data'] as $fd_key => $fdata ) {
									$exp_data->mcqs[ "$m_key" ]['feedback_data'][ $fd_key ] = array(
										'entry' => wpautop( $fdata['entry'] ),
										'rating' => '<i class="ipt-icomoon-' . $setting_to_icon_map[ $fdata['rating'] ] . '"></i>',
									);
									if ( $do_names ) {
										$exp_data->mcqs[ "$m_key" ]['feedback_data'][ $fd_key ]['name'] = $fdata['name'];
									}
									if ( $sensitive_data ) {
										$exp_data->mcqs[ "$m_key" ]['feedback_data'][ $fd_key ]['email'] = ( __( 'anonymous', 'ipt_fsqm_exp' ) !== $fdata['email'] ? '<a href="mailto:' . $fdata['email'] . '">' . $fdata['email'] . '</a>' : $fdata['email'] );
										$exp_data->mcqs[ "$m_key" ]['feedback_data'][ $fd_key ]['id'] = $fdata['id'];
									}
									if ( $do_date ) {
										$exp_data->mcqs[ "$m_key" ]['feedback_data'][ $fd_key ]['date'] = isset( $fdata['date'] ) ? $fdata['date'] : '';
									}
								}
							}
						}
						$return['survey'][ "$m_key" ] = $exp_data->mcqs[ "$m_key" ];
						break;
				}
			}
		}

		// Do the feedbacks
		if ( is_array( $query_elements['freetypes'] ) && ! empty( $query_elements['freetypes'] ) ) {
			foreach ( $query_elements['freetypes'] as $f_key ) {
				$f_key = (int) $f_key;
				if ( ! isset( $util->freetype[ $f_key ] ) || ! isset( $exp_data->freetypes[ $f_key ] ) ) {
					continue;
				}
				$element = $util->freetype[ $f_key ];

				switch ( $element['type'] ) {
					default :
						$definition = $util->get_element_definition( $element );
						if ( isset( $definition['callback_exporter_persistent_sanitizer'] ) && is_callable( $definition['callback_exporter_persistent_sanitizer'] ) ) {
							$return['feedback'][ "$f_key" ] = call_user_func( $definition['callback_exporter_persistent_sanitizer'], $element, $exp_data->freetypes[ "$f_key" ], $f_key, $do_data, $do_names, $do_others, $sensitive_data, $do_date, $util );
						} else {
							$return['feedback'][ "$f_key" ] = $exp_data->freetypes[ "$f_key" ];
						}
						break;
					case 'upload' :
						foreach ( $exp_data->freetypes[ "$f_key" ] as $fu_key => $upload ) {
							$exp_data->freetypes[ "$f_key" ][ $fu_key ] = array(
								'uploads' => $upload['uploads'],
							);
							if ( $do_names ) {
								$exp_data->freetypes[ "$f_key" ][ $fu_key ]['name'] = $upload['name'];
							}
							if ( $sensitive_data ) {
								$exp_data->freetypes[ "$f_key" ][ $fu_key ]['email'] = ( __( 'anonymous', 'ipt_fsqm_exp' ) !== $upload['email'] ? '<a href="mailto:' . $upload['email'] . '">' . $upload['email'] . '</a>' : $upload['email'] );
								$exp_data->freetypes[ "$f_key" ][ $fu_key ]['id'] = $upload['id'];
							}
							if ( $do_date ) {
								$exp_data->freetypes[ "$f_key" ][ $fu_key ]['date'] = $upload['date'];
							}
						}
						$return['feedback'][ "$f_key" ] = $exp_data->freetypes[ "$f_key" ];
						break;
					case 'feedback_large' :
					case 'feedback_small' :
					case 'mathematical' :
						foreach ( $exp_data->freetypes[ "$f_key" ] as $s_key => $sub ) {
							$exp_data->freetypes[ "$f_key" ][ $s_key ] = array(
								'value' => ( $element['type'] == 'mathematical' ? $sub['value'] : wpautop( $sub['value'] ) ),
							);

							if ( $do_names ) {
								$exp_data->freetypes[ "$f_key" ][ $s_key ]['name'] = $sub['name'];
							}
							if ( $sensitive_data ) {
								$exp_data->freetypes[ "$f_key" ][ $s_key ]['email'] = ( __( 'anonymous', 'ipt_fsqm_exp' ) !== $sub['email'] ? '<a href="mailto:' . $sub['email'] . '">' . $sub['email'] . '</a>' : $sub['email'] );
								$exp_data->freetypes[ "$f_key" ][ $s_key ]['phone'] = $sub['phone'];
								$exp_data->freetypes[ "$f_key" ][ $s_key ]['id'] = $sub['id'];
							}
							if ( $do_date ) {
								$exp_data->freetypes[ "$f_key" ][ $s_key ]['date'] = $sub['date'];
							}
						}
						$return['feedback'][ "$f_key" ] = $exp_data->freetypes[ "$f_key" ];
						break;
					case 'gps' :
						foreach ( $exp_data->freetypes[ "$f_key" ] as $g_key => $gps ) {
							$exp_data->freetypes[ "$f_key" ][ $g_key ] = array(
								'map'           => $gps['map'],
								'lat'           => $gps['lat'],
								'long'          => $gps['long'],
								'location_name' => $gps['location_name'],
							);
							if ( $do_names ) {
								$exp_data->freetypes[ "$f_key" ][ $g_key ]['name'] = $gps['name'];
							}
							if ( $sensitive_data ) {
								$exp_data->freetypes[ "$f_key" ][ $g_key ]['email'] = ( __( 'anonymous', 'ipt_fsqm_exp' ) !== $gps['email'] ? '<a href="mailto:' . $gps['email'] . '">' . $gps['email'] . '</a>' : $gps['email'] );
								$exp_data->freetypes[ "$f_key" ][ $g_key ]['phone'] = $gps['phone'];
								$exp_data->freetypes[ "$f_key" ][ $g_key ]['id'] = $gps['id'];
							}
							if ( $do_date ) {
								$exp_data->freetypes[ "$f_key" ][ $g_key ]['date'] = $gps['date'];
							}
						}
						$return['feedback'][ "$f_key" ] = $exp_data->freetypes[ "$f_key" ];
						break;
					case 'feedback_matrix' :
						foreach ( $exp_data->freetypes[ "$f_key" ] as $fm_key => $fmatrix ) {
							$sanitized_matrix = $fmatrix['matrix'];
							array_walk_recursive( $sanitized_matrix, array( __CLASS__, 'autop_array' ) );
							$exp_data->freetypes[ "$f_key" ][ $fm_key ] = array(
								'matrix' => $sanitized_matrix,
							);
							if ( $do_names ) {
								$exp_data->freetypes[ "$f_key" ][ $fm_key ]['name'] = $fmatrix['name'];
							}
							if ( $sensitive_data ) {
								$exp_data->freetypes[ "$f_key" ][ $fm_key ]['email'] = ( __( 'anonymous', 'ipt_fsqm_exp' ) !== $fmatrix['email'] ? '<a href="mailto:' . $fmatrix['email'] . '">' . $fmatrix['email'] . '</a>' : $fmatrix['email'] );
								$exp_data->freetypes[ "$f_key" ][ $fm_key ]['phone'] = $fmatrix['phone'];
								$exp_data->freetypes[ "$f_key" ][ $fm_key ]['id'] = $fmatrix['id'];
							}
							if ( $do_date ) {
								$exp_data->freetypes[ "$f_key" ][ $fm_key ]['date'] = $fmatrix['date'];
							}
						}
						$return['feedback'][ "$f_key" ] = $exp_data->freetypes[ "$f_key" ];
						break;
					case 'signature' :
						foreach ( $exp_data->freetypes[ "$f_key" ] as $fs_key => $sig ) {
							$exp_data->freetypes[ "$f_key" ][ "$fs_key" ] = array(
								'value' => $sig['value'],
							);

							if ( $do_names ) {
								$exp_data->freetypes[ "$f_key" ][ $fs_key ]['name'] = $sig['name'];
							}
							if ( $sensitive_data ) {
								$exp_data->freetypes[ "$f_key" ][ $fs_key ]['email'] = ( __( 'anonymous', 'ipt_fsqm_exp' ) !== $sig['email'] ? '<a href="mailto:' . $sig['email'] . '">' . $sig['email'] . '</a>' : $sig['email'] );
								$exp_data->freetypes[ "$f_key" ][ $fs_key ]['phone'] = $sig['phone'];
								$exp_data->freetypes[ "$f_key" ][ $fs_key ]['id'] = $sig['id'];
							}
							if ( $do_date ) {
								$exp_data->freetypes[ "$f_key" ][ $fs_key ]['date'] = $sig['date'];
							}
						}
						$return['feedback'][ "$f_key" ] = $exp_data->freetypes[ "$f_key" ];
						break;
				}
			}
		}

		// Do the pinfos
		if ( is_array( $query_elements['pinfos'] ) && ! empty( $query_elements['pinfos'] ) ) {
			// Loop through all given pinfos
			foreach ( $query_elements['pinfos'] as $p_key ) {
				$p_key = (int) $p_key;
				if ( ! isset( $util->pinfo[ $p_key ] ) || ! isset( $exp_data->pinfos[ $p_key ] ) ) {
					continue;
				}
				$element = $util->pinfo[ $p_key ];

				switch ( $element['type'] ) {
					default :
						$definition = $util->get_element_definition( $element );
						if ( isset( $definition['callback_exporter_persistent_sanitizer'] ) && is_callable( $definition['callback_exporter_persistent_sanitizer'] ) ) {
							$return['pinfo'][ "$p_key" ] = call_user_func( $definition['callback_exporter_persistent_sanitizer'], $element, $exp_data->pinfos[ "$p_key" ], $p_key, $do_data, $do_names, $do_others, $sensitive_data, $do_date, $util );
						} else {
							$return['pinfo'][ "$p_key" ] = $exp_data->pinfos[ "$p_key" ];
						}
						break;
					// Text types
					case 'f_name' :
					case 'l_name' :
					case 'email' :
					case 'phone' :
					case 'p_name' :
					case 'p_email' :
					case 'p_phone' :
					case 'textinput' :
					case 'textarea' :
					case 'password' :
					case 'keypad' :
					case 'datetime' :
					case 'hidden' :
						foreach ( $exp_data->pinfos[ "$p_key" ] as $t_key => $text ) {
							$exp_data->pinfos[ "$p_key" ][ $t_key ] = array(
								'value' => ( 'datetime' == $element['type'] ? $text['value'] : wpautop( $text['value'] ) ),
							);

							if ( $do_names ) {
								$exp_data->pinfos[ "$p_key" ][ $t_key ]['name'] = $text['name'];
							}
							if ( $sensitive_data ) {
								$exp_data->pinfos[ "$p_key" ][ $t_key ]['email'] = ( __( 'anonymous', 'ipt_fsqm_exp' ) !== $text['email'] ? '<a href="mailto:' . $text['email'] . '">' . $text['email'] . '</a>' : $text['email'] );
								$exp_data->pinfos[ "$p_key" ][ $t_key ]['phone'] = $text['phone'];
								$exp_data->pinfos[ "$p_key" ][ $t_key ]['id'] = $text['id'];
							}
							if ( $do_date ) {
								$exp_data->pinfos[ "$p_key" ][ $t_key ]['date'] = $text['date'];
							}
						}
						$return['pinfo'][ "$p_key" ] = $exp_data->pinfos[ "$p_key" ];
						break;
					case 'address' :
						foreach ( $exp_data->pinfos[ "$p_key" ] as $t_key => $text ) {
							$exp_data->pinfos[ "$p_key" ][ $t_key ] = array(
								'values' => $text['values'],
							);
							if ( $do_names ) {
								$exp_data->pinfos[ "$p_key" ][ $t_key ]['name'] = $text['name'];
							}
							if ( $sensitive_data ) {
								$exp_data->pinfos[ "$p_key" ][ $t_key ]['email'] = ( __( 'anonymous', 'ipt_fsqm_exp' ) !== $text['email'] ? '<a href="mailto:' . $text['email'] . '">' . $text['email'] . '</a>' : $text['email'] );
								$exp_data->pinfos[ "$p_key" ][ $t_key ]['phone'] = $text['phone'];
								$exp_data->pinfos[ "$p_key" ][ $t_key ]['id'] = $text['id'];
							}
							if ( $do_date ) {
								$exp_data->pinfos[ "$p_key" ][ $t_key ]['date'] = $text['date'];
							}
						}
						$return['pinfo'][ "$p_key" ] = $exp_data->pinfos[ "$p_key" ];
						break;
					case 'payment' :
						foreach ( $exp_data->pinfos[ "$p_key" ] as $t_key => $text ) {
							$exp_data->pinfos[ "$p_key" ][ $t_key ] = array(
								'invoice' => $text['invoice'],
								'status' => $text['status'],
								'txn' => $text['txn'],
								'gateway' => $text['gateway'],
								'total' => $text['total'],
							);
							if ( $do_names ) {
								$exp_data->pinfos[ "$p_key" ][ $t_key ]['name'] = $text['name'];
							}
							if ( $sensitive_data ) {
								$exp_data->pinfos[ "$p_key" ][ $t_key ]['email'] = ( __( 'anonymous', 'ipt_fsqm_exp' ) !== $text['email'] ? '<a href="mailto:' . $text['email'] . '">' . $text['email'] . '</a>' : $text['email'] );
								$exp_data->pinfos[ "$p_key" ][ $t_key ]['phone'] = $text['phone'];
								$exp_data->pinfos[ "$p_key" ][ $t_key ]['id'] = $text['id'];
							}
							if ( $do_date ) {
								$exp_data->pinfos[ "$p_key" ][ $t_key ]['date'] = $text['date'];
							}
						}
						$return['pinfo'][ "$p_key" ] = $exp_data->pinfos[ "$p_key" ];
						break;
					case 'p_radio' :
					case 'p_checkbox' :
					case 'p_select' :
						if ( ! $do_others && isset( $exp_data->pinfos[ "$p_key" ]['others_data'] ) ) {
							unset( $exp_data->pinfos[ "$p_key" ]['others_data'] );
						} else {
							if ( isset( $exp_data->pinfos[ "$p_key" ]['others_data'] ) ) {
								foreach ( $exp_data->pinfos[ "$p_key" ]['others_data'] as $o_key => $other ) {
									$exp_data->pinfos[ "$p_key" ]['others_data'][ $o_key ] = array(
										'value' => $other['value'],
									);
									if ( $do_names ) {
										$exp_data->pinfos[ "$p_key" ]['others_data'][ $o_key ]['name'] = $other['name'];
									}
									if ( $sensitive_data ) {
										$exp_data->pinfos[ "$p_key" ]['others_data'][ $o_key ]['email'] = ( __( 'anonymous', 'ipt_fsqm_exp' ) !== $other['email'] ? '<a href="mailto:' . $other['email'] . '">' . $other['email'] . '</a>' : $other['email'] );
										$exp_data->pinfos[ "$p_key" ]['others_data'][ $o_key ]['id'] = $other['id'];
									}
									if ( $do_date ) {
										$exp_data->pinfos[ "$p_key" ]['others_data'][ $o_key ]['date'] = $other['date'];
									}
								}
							}
						}
						$return['pinfo'][ "$p_key" ] = $exp_data->pinfos[ "$p_key" ];
						break;
					case 'guestblog' :
						foreach ( $exp_data->pinfos[ "$p_key" ] as $g_key => $gblog ) {
							$exp_data->pinfos[ "$p_key" ][ $g_key ] = array(
								'value' => $gblog['value'],
								'title' => $gblog['title'],
								'taxonomy' => $gblog['taxonomy'],
							);
							if ( $do_names ) {
								$exp_data->pinfos[ "$p_key" ][ $g_key ]['name'] = $gblog['name'];
							}
							if ( $sensitive_data ) {
								$exp_data->pinfos[ "$p_key" ][ $g_key ]['email'] = ( __( 'anonymous', 'ipt_fsqm_exp' ) !== $gblog['email'] ? '<a href="mailto:' . $gblog['email'] . '">' . $gblog['email'] . '</a>' : $gblog['email'] );
								$exp_data->pinfos[ "$p_key" ][ $g_key ]['phone'] = $gblog['phone'];
								$exp_data->pinfos[ "$p_key" ][ $g_key ]['id'] = $gblog['id'];
							}
							if ( $do_date ) {
								$exp_data->pinfos[ "$p_key" ][ $g_key ]['date'] = $gblog['date'];
							}
						}
						$return['pinfo'][ "$p_key" ] = $exp_data->pinfos[ "$p_key" ];
						break;
					case 'repeatable' :
						foreach ( $exp_data->pinfos[ "$p_key" ] as $r_key => $item_row ) {
							$pinfo_text = array(
								'value' => '<table class="data-table"><tbody>',
							);
							$i = 1;
							foreach ( $item_row['values'] as $i_key => $items ) {
								$rowspan = count( (array) $element['settings']['group'] ) + 1;
								$pinfo_text['value'] .= '<tr><th rowspan="' . $rowspan . '">' . sprintf( _x( '#%d', 'eform-repetable-heading', 'ipt_fsqm' ), $i++ ) . '</th></tr>';
								foreach ( (array) $element['settings']['group'] as $g_key => $group ) {
									$pinfo_text['value'] .= '<tr><th>' . $group['title'] . '</th><td>';

									if ( isset( $items[ $g_key ] ) ) {
										switch ( $group['type'] ) {
											case 'radio' :
												$pinfo_text['value'] .= str_replace( '__', ' ', $items[ $g_key ] );
												break;
											case 'checkbox' :
											case 'select' :
											case 'select_multiple' :
												$options = array();
												foreach ( (array) $items[ $g_key ] as $op ) {
													$options[] = str_replace( '__', ' ', $op );
												}
												$pinfo_text['value'] .= implode( '<br />', $options );
												break;
											case 'text' :
											case 'phone' :
											case 'url' :
											case 'email' :
											case 'number' :
											case 'integer' :
											case 'personName' :
											case 'password' :
											case 'textarea' :
											case 'date' :
											case 'time' :
											case 'datetime' :
												$pinfo_text['value'] .= $items[ $g_key ];
												break;
										}
									}

									$pinfo_text['value'] .= '</td></tr>';
								}
							}

							$pinfo_text['value'] .= '</tbody></table>';

							if ( $do_names ) {
								$pinfo_text['name'] = $item_row['name'];
							}
							if ( $sensitive_data ) {
								$pinfo_text['email'] = $item_row['email'];
								$pinfo_text['phone'] = $item_row['phone'];
								$pinfo_text['id'] = $item_row['id'];
							}
							if ( $do_date ) {
								$pinfo_text['date'] = $item_row['date'];
							}
							$return['pinfo'][ "$p_key" ][] = $pinfo_text;
						}
						break;
				}
			}
		}

		// Echo the json and exit
		$return['survey'] = (object) $return['survey'];
		$return['feedback'] = (object) $return['feedback'];
		$return['pinfo'] = (object) $return['pinfo'];
		@header( 'Content-Type: application/json; charset=' . get_option( 'blog_charset' ) );

		echo json_encode( (object) $return );
		die(); // die die die
	}

	/*==========================================================================
	 * Callback for Form Delete from FSQM Pro
	 *========================================================================*/
	public static function sync_form_delete( $form_ids ) {
		global $ipt_fsqm_exp_info, $wpdb;
		if ( ! is_array( $form_ids ) ) {
			$form_ids = (array) $form_ids;
		}

		if ( empty( $form_ids ) ) {
			return false;
		}

		$form_ids = array_map( 'intval', $form_ids );
		$form_ids_in = implode( ',', $form_ids );
		$report_ids = $wpdb->get_col( "SELECT id FROM {$ipt_fsqm_exp_info['exp_table']} WHERE form_id IN ({$form_ids_in})" );
		$raw_ids = $wpdb->get_col( "SELECT id FROM {$ipt_fsqm_exp_info['raw_table']} WHERE form_id IN ({$form_ids_in})" );

		if ( null == $report_ids || empty( $report_ids ) || ! is_array( $report_ids ) ) {
			$report_ids = array();
		}

		if ( null == $raw_ids || empty( $raw_ids ) || ! is_array( $raw_ids ) ) {
			$raw_ids = array();
		}

		self::delete_exp( $report_ids );
		self::delete_raw( $raw_ids );
	}

	/*==========================================================================
	 * Database Abstractions
	 *========================================================================*/

	public static function get_exports() {
		global $wpdb, $ipt_fsqm_exp_info, $ipt_fsqm_info;
		return $wpdb->get_results( "SELECT e.created created, f.name name, e.id id FROM {$ipt_fsqm_exp_info['exp_table']} e LEFT JOIN {$ipt_fsqm_info['form_table']} f ON e.form_id = f.id ORDER BY e.id DESC" );
	}

	public static function get_exp_data( $exp_id ) {
		global $wpdb, $ipt_fsqm_exp_info;
		$data = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$ipt_fsqm_exp_info['exp_table']} WHERE id = %d", $exp_id ) );
		if ( null == $data ) {
			return false;
		}
		$data->freetypes = (array) maybe_unserialize( $data->freetypes );
		$data->mcqs = (array) maybe_unserialize( $data->mcqs );
		$data->pinfos = (array) maybe_unserialize( $data->pinfos );

		// Now we do a little compatibility check
		if ( '' == $data->meta ) {
			$meta = array(
				'mcqs' => array_keys( $data->mcqs ),
				'freetypes' => array_keys( $data->freetypes ),
				'pinfos' => array(),
				'filters' => array(),
				'settings' => array(
					'report' => array(
						'mcq',
						'freetype',
					),
				),
			);
			$data->meta = $meta;
		} else {
			$data->meta = (array) maybe_unserialize( $data->meta );
		}
		return $data;
	}

	public static function get_exp_chart_elements( $exp_id ) {
		// Prepare an empty return
		$return = array(
			'mcqs' => array(),
			'pinfos' => array(),
		);

		// Get the export data
		$exp_data = self::get_exp_data( $exp_id );

		// If it is false, then simply return empty array
		if ( false === $exp_data ) {
			return $return;
		}

		// Create a dummy form for checking element type
		$form = new IPT_FSQM_Form_Elements_Base( $exp_data->form_id );

		// If invalid form, then also return the empty array
		if ( is_null( $form->form_id ) ) {
			return $return;
		}

		// pinfo types for charts
		$pinfo_charts = IPT_FSQM_Form_Elements_Static::get_pinfo_chart_elements();

		// Iterate through all mcqs
		foreach ( $form->mcq as $m_key => $mcq ) {
			// If not present in the metadata, then skip
			if ( ! in_array( (int) $m_key, $exp_data->meta['mcqs'] ) ) {
				continue;
			}

			$return['mcqs'][] = array(
				'key' => $m_key,
				'type' => $mcq['type'],
				'title' => $mcq['title'],
			);
		}

		// Iterate through all pinfo elements
		foreach ( $form->pinfo as $p_key => $pinfo ) {
			// If not present in the metadata, then skip
			if ( ! in_array( (int) $p_key, $exp_data->meta['pinfos'] ) ) {
				continue;
			}

			if ( in_array( $pinfo['type'], $pinfo_charts ) ) {
				$return['pinfos'][] = array(
					'key' => $p_key,
					'type' => $pinfo['type'],
					'title' => $pinfo['title'],
				);
			}
		}

		return apply_filters( 'ipt_fsqm_exp_form_chart_elements', $return, $exp_id, $pinfo_charts );
	}

	public static function delete_exp( $ids = array() ) {
		global $ipt_fsqm_exp_info, $wpdb;
		if ( ! is_array( $ids ) ) {
			$ids = (array) $ids;
		}

		if ( empty( $ids ) ) {
			return false;
		}

		do_action( 'ipt_fsqm_exp_exports_deleted', $ids );

		// Delete entries from database
		$ids = array_map( 'intval', $ids );
		$delete_ids = implode( ',', $ids );
		$return = $wpdb->query( "DELETE FROM {$ipt_fsqm_exp_info['exp_table']} WHERE id IN ({$delete_ids})" );

		// Now loop through and delete individual files
		$file_extensions = self::get_available_export_formats();
		$wp_path = wp_upload_dir();
		$path = $wp_path['basedir'] . '/fsqm-exp-reports';

		foreach ( $ids as $id ) {
			foreach ( $file_extensions as $type ) {
				$file_path = $path . '/ipt-fsqm-exp-' . $id . '.' . $type;

				// Delete the zip file
				if ( file_exists( $file_path . '.zip' ) ) {
					@unlink( $file_path . '.zip' );
				}

				// Delete the directory
				if ( file_exists( $file_path ) ) {
					if ( is_dir( $file_path ) ) {
						self::delTree( $file_path );
					} else {
						@unlink( $file_path );
					}
				}
			}
		}

		// And we are done
		return $return;
	}

	public static function get_raws() {
		global $wpdb, $ipt_fsqm_exp_info, $ipt_fsqm_info;
		return $wpdb->get_results( "SELECT r.id id, r.created created, r.start_date, start_date, r.end_date end_date, f.name name FROM {$ipt_fsqm_exp_info['raw_table']} r LEFT JOIN {$ipt_fsqm_info['form_table']} f ON r.form_id = f.id" );
	}

	public static function get_raw_data( $raw_id ) {
		global $wpdb, $ipt_fsqm_exp_info;
		$data = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$ipt_fsqm_exp_info['raw_table']} WHERE id = %d", $raw_id ) );
		if ( null == $data ) {
			return false;
		}
		$data->freetype = maybe_unserialize( $data->freetype );
		$data->mcq = maybe_unserialize( $data->mcq );
		$data->pinfo = maybe_unserialize( $data->pinfo );
		return $data;
	}

	public static function delete_raw( $ids = array() ) {
		global $wpdb, $ipt_fsqm_exp_info;
		if ( ! is_array( $ids ) ) {
			$ids = (array) $ids;
		}

		if ( empty( $ids ) ) {
			return false;
		}

		do_action( 'ipt_fsqm_exp_raws_deleted', $ids );

		// Prepare delete entries from database
		$ids = array_map( 'intval', $ids );
		$delete_ids = implode( ',', $ids );


		// Now loop through and delete individual files
		$entries = $wpdb->get_results( "SELECT id, form_id FROM {$ipt_fsqm_exp_info['raw_table']} WHERE id IN ({$delete_ids})" );
		$wp_path = wp_upload_dir();
		$path = $wp_path['basedir'] . '/fsqm-exp-raw';

		foreach ( $entries as $entry ) {
			$file_path = $path . '/raw-' . $entry->form_id . '-' . $entry->id . '.csv';

			// Delete the zip file
			if ( file_exists( $file_path ) ) {
				@unlink( $file_path );
			}

			$xlsx_path = $path . '/raw-' . $entry->form_id . '-' . $entry->id . '.xlsx';

			// Delete
			if ( file_exists( $xlsx_path ) ) {
				@unlink( $xlsx_path );
			}
		}

		// Delete the entries
		$return = $wpdb->query( "DELETE FROM {$ipt_fsqm_exp_info['raw_table']} WHERE id IN ({$delete_ids})" );

		// And we are done
		return $return;
	}

	/*==========================================================================
	 * Exporter functions
	 *========================================================================*/

	public static function ipt_fsqm_exp_export_report_to_file_cb() {
		global $wpdb, $ipt_fsqm_exp_info;
		if ( !isset( $_REQUEST['id'] ) || !isset( $_REQUEST['_wpnonce'] ) ) {
			wp_die( __( 'Cheatin&#8217; uh?' ) );
		}

		if( !current_user_can( 'manage_feedback' ) ) {
			wp_die( __( 'Cheatin&#8217; uh?' ) );
		}

		$id = (int) $_REQUEST['id'];

		if ( !wp_verify_nonce( $_REQUEST['_wpnonce'], 'ipt_fsqm_exp_export_report_to_file_' . $id ) ) {
			wp_die( __( 'Cheatin&#8217; uh?' ) );
		}

		require_once IPT_FSQM_EXP_Loader::$abs_path . '/classes/class-ipt-fsqm-exp-form-elements-export-report.php';
		$report = new IPT_FSQM_EXP_Form_Elements_Export_Report( $id );
		$report->stream_export( $_REQUEST['type'] );
		die();
	}

	/**
	 * Export Report calculation
	 *
	 * @global  wpdb $wpdb
	 * @global array $ipt_fsqm_exp_info
	 * @global array $ipt_fsqm_info
	 * @return void
	 */
	public static function ipt_fsqm_exp_export_report_cb() {
		global $wpdb, $ipt_fsqm_exp_info, $ipt_fsqm_info, $ipt_fsqm_settings;

		// Unslash the variable
		$post = wp_unslash( $_POST );

		// Get the variables
		$doing = isset( $post['doing'] ) ? (int) $post['doing'] : 0;
		$form_id = isset( $post['form_id'] ) ? (int) $post['form_id'] : 0;
		$exp_id = isset( $post['exp_id'] ) ? (int) $post['exp_id'] : 0;

		// Check for nonce
		if ( ! wp_verify_nonce( $_POST['wpnonce'], 'ipt_fsqm_exp_export_report_' . $form_id ) ) {
			die( __( 'Cheatin&#8217; uh?' ) );
		}

		// First test the form_id
		if ( null == $wpdb->get_var( $wpdb->prepare( "SELECT name FROM {$ipt_fsqm_info['form_table']} WHERE id = %d", $form_id ) ) ) {
			die( __( 'Cheatin&#8217; uh?' ) );
		}

		// If hitting for the first time
		if ( 0 == $doing && 0 == $exp_id ) {
			// Get the variables from JS for the first time
			$mcqs = isset( $post['mcqs'] ) ? $post['mcqs'] : array();
			$freetypes = isset( $post['freetypes'] ) ? $post['freetypes'] : array();
			$pinfos = isset( $post['pinfos'] ) ? $post['pinfos'] : array();
			$filters = isset( $post['filters'] ) ? $post['filters'] : array();
			$settings = isset( $post['settings'] ) ? $post['settings'] : array();
			// Setup the meta
			$meta = array(
				'mcqs' => $mcqs,
				'freetypes' => $freetypes,
				'pinfos' => $pinfos,
				'filters' => $filters,
				'settings' => $settings,
			);
			// Create a new entry
			$exp_data_new = array(
				'form_id' => $form_id,
				'created' => current_time( 'mysql' ),
				'complete' => 0,
				'start_date' => isset( $filters['custom_date'] ) && 'true' == $filters['custom_date'] && isset( $filters['custom_date_start'] ) && '' != $filters['custom_date_start'] ? $filters['custom_date_start'] : '0000-00-00 00:00:00',
				'end_date' => isset( $filters['custom_date'] ) && 'true' == $filters['custom_date'] && isset( $filters['custom_date_end'] ) && '' != $filters['custom_date_end'] ? $filters['custom_date_end'] : '0000-00-00 00:00:00',
				'mcqs' => maybe_serialize( array() ),
				'freetypes' => maybe_serialize( array() ),
				'pinfos' => maybe_serialize( array() ),
				'meta' => maybe_serialize( $meta ),
			);
			$wpdb->insert( $ipt_fsqm_exp_info['exp_table'], $exp_data_new, array(
				'%d', '%s', '%d', '%s', '%s', '%s', '%s', '%s', '%s'
			) );
			$exp_id = $wpdb->insert_id;
		// If successive calls
		} else {
			if ( ! wp_verify_nonce( $_POST['exp_nonce'], 'ipt_fsqm_exp_export_report_' . $form_id . '_exp_' . $exp_id ) ) {
				die( __( 'Cheatin&#8217; uh?' ) );
			}
			if ( null == $wpdb->get_var( $wpdb->prepare( "SELECT form_id FROM {$ipt_fsqm_exp_info['exp_table']} WHERE id = %d", $exp_id ) ) ) {
				die( __( 'Cheatin&#8217; uh?' ) );
			}
		}
		// Get the DB entry
		$exp_row = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$ipt_fsqm_exp_info['exp_table']} WHERE id = %d", $exp_id ) );
		// Prepare the expdata
		$exp_data = array(
			'mcqs' => (array) maybe_unserialize( $exp_row->mcqs ),
			'freetypes' => (array) maybe_unserialize( $exp_row->freetypes ),
			'pinfos' => (array) maybe_unserialize( $exp_row->pinfos ),
			'start_date' => $exp_row->start_date,
			'end_date' => $exp_row->end_date,
		);
		// Override the post variables from the ones stored in db
		// This is for consistency
		$meta = maybe_unserialize( $exp_row->meta );
		$mcqs = $meta['mcqs'];
		$freetypes = $meta['freetypes'];
		$pinfos = $meta['pinfos'];
		$filters = $meta['filters'];
		$settings = $meta['settings'];

		// Prepare the link
		$return = array(
			'type' => 'success',
			'done' => 0,
			'wpnonce' => wp_create_nonce( 'ipt_fsqm_exp_export_report_' . $form_id ),
			'exp_nonce' => wp_create_nonce( 'ipt_fsqm_exp_export_report_' . $form_id . '_exp_' . $exp_id ),
			'exp_id' => $exp_id,
			'links' => array(),
			'debug_info' => array(),
		);

		// Populate links to buttons
		$available_export_formats = self::get_available_export_formats();
		foreach ( $available_export_formats as $format ) {
			$return['links'][$format] = admin_url( 'admin-ajax.php?action=ipt_fsqm_exp_export_report_to_file&type=' . $format . '&id=' . $exp_id . '&_wpnonce=' . wp_create_nonce( 'ipt_fsqm_exp_export_report_to_file_' . $exp_id ) );
		}
		$return['links']['persistent_report'] = admin_url( 'admin.php?page=ipt_fsqm_exp_view_all_reports&id=' . $exp_id );

		// Calculate the number of data to fetch
		$per_page = 15;
		if ( isset( $settings['load'] ) ) {
			switch ( $settings['load'] ) {
				case '1' :
					$per_page = 30;
					break;
				case '2' :
					$per_page = 50;
			}
		}
		$debug_info['load'] = $settings['load'];
		$debug_info['per_page'] = $per_page;
		$debug_info['doing'] = $doing;

		// Set the header for JSON content type
		@header( 'Content-Type: application/json; charset=' . get_option( 'blog_charset' ) );

		// Add the filters
		$where = '';
		$where_arr = array();
		// Check the date
		if ( $exp_data['start_date'] != '0000-00-00 00:00:00' ) {
			$where_arr[] = $wpdb->prepare( 'date >= %s', date( 'Y-m-d H:i:s', strtotime( $exp_data['start_date'] ) ) );
		}
		if ( $exp_data['end_date'] != '0000-00-00 00:00:00' ) {
			$where_arr[] = $wpdb->prepare( 'date <= %s', date( 'Y-m-d H:i:s', strtotime( $exp_data['end_date'] ) ) );
		}

		// Check for score filter
		if ( isset( $filters['score'] ) && is_array( $filters['score'] ) ) {
			// Minimum filter
			if ( isset( $filters['score']['min'] ) && '' != $filters['score']['min'] ) {
				$where_arr[] = $wpdb->prepare( 'score >= %f', $filters['score']['min'] );
			}
			// Maximum filter
			if ( isset( $filters['score']['max'] ) && '' != $filters['score']['max'] ) {
				$where_arr[] = $wpdb->prepare( 'score <= %f', $filters['score']['max'] );
			}
		}

		// Check for url_track
		if ( isset( $filters['url_track'] ) && is_array( $filters['url_track'] ) ) {
			// Omit if it is just all
			if ( ! in_array( '', $filters['url_track'] ) ) {
				$url_track_where = array();

				foreach ( $filters['url_track'] as $ut ) {
					$url_track_where[] = $wpdb->prepare( 'url_track = %s', $ut );
				}

				if ( ! empty( $url_track_where ) ) {
					$where_arr[] = '( ' . implode( ' OR ', $url_track_where ) . ' )';
				}
			}
		}

		// Check for user meta
		$user_ids = array();
		if ( isset( $filters['meta'] ) && '' != $filters['meta'] ) {
			$get_users_args = array(
				'meta_key' => $filters['meta'],
				'fields' => 'ID',
			);

			if ( isset( $filters['mvalue'] ) && '' != $filters['mvalue'] ) {
				$get_users_args['meta_value'] = $filters['mvalue'];
			}
			$users_from_meta = get_users( $get_users_args );
			if ( ! empty( $users_from_meta ) ) {
				$user_ids = array_merge( $user_ids, $users_from_meta );
			} else {
				$user_ids = array( '-1' );
			}
		}

		// Check for user_id
		if ( isset( $filters['user_id'] ) && is_array( $filters['user_id'] ) ) {
			// Omit if it is just all
			if ( ! in_array( '', $filters['user_id'] ) ) {
				$user_ids = array_merge( $user_ids, $filters['user_id'] );
			}
		}

		// Add the user id to the query
		if ( ! empty( $user_ids ) ) {
			$user_ids = array_unique( $user_ids );
			$user_ids = array_map( 'intval', $user_ids );
			$where_arr[] = 'user_id IN (' . implode( ',', $user_ids ) . ')';
		}

		if ( ! empty( $where_arr ) ) {
			$where .= ' AND ' . implode( ' AND ', $where_arr );
		}

		// Get data ids
		$data_ids = $wpdb->get_col( $wpdb->prepare( "SELECT id FROM {$ipt_fsqm_info['data_table']} WHERE form_id = %d{$where} ORDER BY id ASC LIMIT %d,%d", $form_id, $doing * $per_page, $per_page ) );

		// Get total number of data
		$total = $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(id) FROM {$ipt_fsqm_info['data_table']} WHERE form_id = %d{$where}", $form_id ) );

		// If no data ID found, then just return
		if ( empty( $data_ids ) ) {
			$return['done'] = 100;
			$wpdb->update( $ipt_fsqm_exp_info['exp_table'], array(
					'complete' => 1,
				), array(
					'id' => $exp_id,
				), '%d', '%d' );
			echo json_encode( (object) $return );
			die();
		}

		// Some helper variables
		$date_formats = array(
			'yy-mm-dd' => 'Y-m-d',
			'mm/dd/yy' => 'm/d/Y',
			'dd.mm.yy' => 'd.m.Y',
			'dd-mm-yy' => 'd-m-Y',
		);
		$time_formats = array(
			'HH:mm:ss' => 'H:i:s',
			'hh:mm:ss TT' => 'h:i:s A',
		);

		foreach ( $data_ids as $data_id ) {
			$data = new IPT_FSQM_Form_Elements_Data( $data_id );

			$data_formatted_date = date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), strtotime( $data->data->date ) );
			$data_formatted_email = $data->data->email == '' ? __( 'anonymous', 'ipt_fsqm_exp' ) : $data->data->email;
			$data_formatted_name = ( $data->data->f_name == '' && $data->data->l_name == '' ) ? __( 'anonymous', 'ipt_fsqm_exp' ) : $data->data->f_name . ' ' . $data->data->l_name;

			// blacklist the conditionally hidden elements
			$data->blacklist_conditional_hiddens();

			// Do the survey
			if ( is_array( $mcqs ) && !empty( $mcqs ) ) {
				foreach ( $mcqs as $m_key ) {
					// Cast to a true integer, because it is stored as string (string '1' not int 1)
					$m_key = (int) $m_key;
					// Check if it exists
					if ( ! isset( $data->mcq[ $m_key ] ) || ! isset( $data->data->mcq[ $m_key ] ) ) {
						continue;
					}
					// Get the element
					$element = $data->mcq[ $m_key ];

					if ( isset( $data->conditional_hidden_blacklist[$element['m_type']] ) ) {
						if ( in_array( (string) $m_key, $data->conditional_hidden_blacklist[$element['m_type']] ) ) {
							continue;
						}
					}

					switch ( $data->mcq[$m_key]['type'] ) {
						default :
							$definition = $data->get_element_definition( $element );
							if ( isset( $definition['callback_report_calculator'] ) && is_callable( $definition['callback_report_calculator'] ) ) {
								if ( ! isset( $exp_data['mcqs']["$m_key"] ) ) {
									$exp_data['mcqs']["$m_key"] = array();
								}
								$exp_data['mcqs']["$m_key"] = call_user_func( $definition['callback_report_calculator'], $element, $data->data->mcq[$m_key], $m_key, $exp_data['mcqs']["$m_key"]  );
							}
							break;
						case 'radio' :
						case 'checkbox' :
						case 'select' :
						case 'thumbselect' :
							if ( ! isset( $exp_data['mcqs']["$m_key"] ) ) {
								$exp_data['mcqs']["$m_key"] = array();
								$exp_data['mcqs']["$m_key"]['others_data'] = array();
							}
							if ( ! isset( $data->data->mcq[$m_key]['options'] ) || empty( $data->data->mcq[$m_key]['options'] ) ) {
								continue 2;
							}
							foreach ( $data->data->mcq[$m_key]['options'] as $o_key ) {
								$exp_data['mcqs']["$m_key"]["$o_key"] = isset( $exp_data['mcqs']["$m_key"]["$o_key"] ) ? $exp_data['mcqs']["$m_key"]["$o_key"] + 1 : 1;
							}
							if ( ! empty( $data->data->mcq[$m_key]['others'] ) ) {
								$exp_data['mcqs']["$m_key"]['others_data'][] = array(
									'value' => esc_textarea( $data->data->mcq[$m_key]['others'] ),
									'name' => $data_formatted_name,
									'email' => $data_formatted_email,
									'id' => $data->data_id,
									'date' => $data_formatted_date,
								);
							}
							break;
						case 'smileyrating' :
							$setting_to_icon_map = array(
								'frown' => 'angry2',
								'sad' => 'sad2',
								'neutral' => 'neutral2',
								'happy' => 'smiley2',
								'excited' => 'happy2',
							);
							if ( ! isset( $exp_data['mcqs']["$m_key"] ) ) {
								$exp_data['mcqs']["$m_key"] = array(
									'frown' => 0,
									'sad' => 0,
									'neutral' => 0,
									'happy' => 0,
									'excited' => 0,
								);
								$exp_data['mcqs']["$m_key"]['feedback_data'] = array();
							}
							if ( $data->data->mcq[ $m_key ]['option'] == '' ) {
								continue 2;
							}

							if ( isset( $exp_data['mcqs']["$m_key"][$data->data->mcq[$m_key]['option']] ) ) {
								$exp_data['mcqs']["$m_key"][$data->data->mcq[$m_key]['option']]++;
							}
							if ( isset( $data->data->mcq[$m_key]['feedback'] ) && $data->data->mcq[$m_key]['feedback'] != '' ) {
								$exp_data['mcqs']["$m_key"]['feedback_data'][] = array(
									'entry' => esc_textarea( $data->data->mcq[$m_key]['feedback'] ),
									'rating' => $data->data->mcq[$m_key]['option'],
									'name' => $data_formatted_name,
									'email' => $data_formatted_email,
									'id' => $data->data_id,
									'date' => $data_formatted_date,
								);
							}
							break;
						case 'likedislike' :
							if ( ! isset( $exp_data['mcqs']["$m_key"] ) ) {
								$exp_data['mcqs']["$m_key"] = array(
									'like' => 0,
									'dislike' => 0,
								);
								$exp_data['mcqs']["$m_key"]['feedback_data'] = array();
							}
							if ( $data->data->mcq[$m_key]['value'] == '' ) {
								continue 2;
							}
							if ( isset( $exp_data['mcqs']["$m_key"][$data->data->mcq[$m_key]['value']] ) ) {
								$exp_data['mcqs']["$m_key"][$data->data->mcq[$m_key]['value']]++;
							}
							if ( isset( $data->data->mcq[$m_key]['feedback'] ) && $data->data->mcq[$m_key]['feedback'] != '' ) {
								$exp_data['mcqs']["$m_key"]['feedback_data'][] = array(
									'entry' => esc_textarea( $data->data->mcq[$m_key]['feedback'] ),
									'rating' => $data->data->mcq[$m_key]['value'],
									'name' => $data_formatted_name,
									'email' => $data_formatted_email,
									'id' => $data->data_id,
									'date' => $data_formatted_date,
								);
							}
							break;
						case 'slider' :
							if ( ! isset( $exp_data['mcqs']["$m_key"] ) ) {
								$exp_data['mcqs']["$m_key"] = array();
							}
							if ( '' == $data->data->mcq[$m_key]['value'] ) {
								continue 2;
							}
							$exp_data['mcqs']["$m_key"]["{$data->data->mcq[$m_key]['value']}"] = isset( $exp_data['mcqs']["$m_key"]["{$data->data->mcq[$m_key]['value']}"] ) ? $exp_data['mcqs']["$m_key"]["{$data->data->mcq[$m_key]['value']}"] + 1 : 1;
							break;
						case 'range' :
							if ( ! isset( $exp_data['mcqs']["$m_key"] ) ) {
								$exp_data['mcqs']["$m_key"] = array();
							}
							if ( empty( $data->data->mcq[$m_key]['values'] ) ) {
								continue 2;
							}
							$key = "{$data->data->mcq[$m_key]['values']['min']},{$data->data->mcq[$m_key]['values']['max']}";
							$exp_data['mcqs']["$m_key"][$key] = isset( $exp_data['mcqs']["$m_key"][$key] ) ? $exp_data['mcqs']["$m_key"][$key] + 1 : 1;
							break;
						case 'spinners' :
						case 'grading' :
						case 'starrating' :
						case 'scalerating' :
							if ( ! isset( $exp_data['mcqs']["$m_key"] ) ) {
								$exp_data['mcqs']["$m_key"] = array();
							}
							if ( empty( $data->data->mcq[$m_key]['options'] ) ) {
								continue 2;
							}

							foreach ( $data->mcq[$m_key]['settings']['options'] as $o_key => $o_val ) {
								if ( ! isset( $exp_data['mcqs']["$m_key"]["$o_key"] ) ) {
									$exp_data['mcqs']["$m_key"]["$o_key"] = array();
								}
								if ( !isset( $data->data->mcq[$m_key]['options'][$o_key] ) ) {
									continue;
								}
								if ( is_array( $data->data->mcq[$m_key]['options'][$o_key] ) ) {
									$key = $data->data->mcq[$m_key]['options'][$o_key]['min'] . ',' . $data->data->mcq[$m_key]['options'][$o_key]['max'];
								} else {
									$key = (string) $data->data->mcq[$m_key]['options'][$o_key];
								}

								if ( $key == '' ) {
									continue;
								}

								$exp_data['mcqs']["$m_key"]["$o_key"][$key] = isset( $exp_data['mcqs']["$m_key"]["$o_key"][$key] ) ? $exp_data['mcqs']["$m_key"]["$o_key"][$key] + 1 : 1;
							}
							break;
						case 'matrix_dropdown' :
							if ( ! isset( $exp_data['mcqs']["$m_key"] ) ) {
								$exp_data['mcqs']["$m_key"] = array();
							}
							if ( empty( $data->data->mcq[$m_key]['rows'] ) ) {
								continue 2;
							}
							$options_array = array();
							foreach ( $data->mcq[$m_key]['settings']['options'] as $o_key => $op ) {
								$options_array["$o_key"] = 0;
							}
							foreach ( (array) $data->data->mcq[$m_key]['rows'] as $r_key => $columns ) {
								if ( ! isset( $exp_data['mcqs']["$m_key"]["$r_key"] ) ) {
									$exp_data['mcqs']["$m_key"]["$r_key"] = array();
								}
								foreach ( (array) $columns as $c_key => $o_key ) {
									if ( ! isset( $exp_data['mcqs']["$m_key"]["$r_key"]["$c_key"] ) ) {
										$exp_data['mcqs']["$m_key"]["$r_key"]["$c_key"] = $options_array;
									}

									if ( is_array( $o_key ) ) {
										foreach( $o_key as $option_key ) {
											if ( isset( $exp_data['mcqs']["$m_key"]["$r_key"]["$c_key"]["$option_key"] ) ) {
												$exp_data['mcqs']["$m_key"]["$r_key"]["$c_key"]["$option_key"]++;
											}
										}
									} else {
										if ( isset( $exp_data['mcqs']["$m_key"]["$r_key"]["$c_key"]["$o_key"] ) ) {
											$exp_data['mcqs']["$m_key"]["$r_key"]["$c_key"]["$o_key"]++;
										}
									}
								}
							}
							break;
						case 'matrix' :
							if ( ! isset( $exp_data['mcqs']["$m_key"] ) ) {
								$exp_data['mcqs']["$m_key"] = array();
							}
							if ( empty( $data->data->mcq[$m_key]['rows'] ) ) {
								continue 2;
							}
							foreach ( $data->data->mcq[$m_key]['rows'] as $r_key => $columns ) {
								if ( !isset( $exp_data['mcqs']["$m_key"]["$r_key"] ) ) {
									$exp_data['mcqs']["$m_key"]["$r_key"] = array();
								}
								foreach ( $columns as $c_key ) {
									$exp_data['mcqs']["$m_key"]["$r_key"]["$c_key"] = isset( $exp_data['mcqs']["$m_key"]["$r_key"]["$c_key"] ) ? $exp_data['mcqs']["$m_key"]["$r_key"]["$c_key"] + 1 : 1;
								}
							}
							break;
						case 'toggle' :
							if ( ! isset( $exp_data['mcqs']["$m_key"] ) ) {
								$exp_data['mcqs']["$m_key"] = array(
									'on' => 0,
									'off' => 0,
								);
							}
							if ( $data->data->mcq[$m_key]['value'] == false ) {
								$exp_data['mcqs']["$m_key"]['off']++;
							} else {
								$exp_data['mcqs']["$m_key"]['on']++;
							}
							break;
						case 'sorting' :
							if ( ! isset( $exp_data['mcqs']["$m_key"] ) ) {
								$exp_data['mcqs']["$m_key"] = array(
									'preset' => 0,
									'other' => 0,
									'orders' => array(),
								);
							}
							if ( empty( $data->data->mcq[$m_key]['order'] ) ) {
								continue 2;
							}
							$correct_order = implode( '-', array_keys( $data->mcq[$m_key]['settings']['options'] ) );
							$user_order = implode( '-', $data->data->mcq[$m_key]['order'] );
							if ( $correct_order == $user_order ) {
								$exp_data['mcqs']["$m_key"]['preset']++;
							} else {
								$exp_data['mcqs']["$m_key"]['other']++;
							}
							$exp_data['mcqs']["$m_key"]['orders'][$user_order] = isset( $exp_data['mcqs']["$m_key"]['orders'][$user_order] ) ? $exp_data['mcqs']["$m_key"]['orders'][$user_order] + 1 : 1;
					}
				}
			}

			// Do the Feedback
			if ( is_array( $freetypes ) && ! empty( $freetypes ) ) {
				foreach ( $freetypes as $f_key ) {
					$f_key = (int) $f_key;
					// Check if it exists
					if ( ! isset( $data->freetype[ $f_key ] ) || ! isset( $data->data->freetype[ $f_key ] ) ) {
						continue;
					}
					// Get the element
					$element = $data->freetype[ $f_key ];

					if ( isset( $data->conditional_hidden_blacklist[$element['m_type']] ) ) {
						if ( in_array( (string) $f_key, $data->conditional_hidden_blacklist[$element['m_type']] ) ) {
							continue;
						}
					}

					switch ( $element['type'] ) {
						default :
							$definition = $data->get_element_definition( $element );
							if ( isset( $definition['callback_report_calculator'] ) && is_callable( $definition['callback_report_calculator'] ) ) {
								if ( isset( $exp_data['freetypes']["$f_key"] ) ) {
									$exp_data['freetypes']["$f_key"] = array();
								}
								$exp_data['freetypes']["$f_key"] = call_user_func( $definition['callback_report_calculator'], $element, $data->data->freetype[$f_key], $f_key, $exp_data['freetypes']["$f_key"] );
							}
							break;
						case 'upload' :
							if ( ! isset( $exp_data['freetypes']["$f_key"] ) ) {
								$exp_data['freetypes']["$f_key"] = array();
							}

							// Init the uploader class
							$uploader = new IPT_FSQM_Form_Elements_Uploader( $data->form_id, $f_key );
							$uploads = $uploader->get_uploads( $data->data_id );

							// Loop through all uploads and save the meta
							$upload_array = array();

							if ( ! empty( $uploads ) ) {
								foreach ( $uploads as $upload ) {
									if ( '' == $upload['guid'] ) {
										continue;
									}
									$upload_array[] = array(
										'guid'      => $upload['guid'],
										'thumb_url' => $upload['thumb_url'],
										'name'      => $upload['name'] . ' (' . $upload['mime_type'] . ' )',
										'filename'  => $upload['filename'],
									);
								}
							}

							$exp_data['freetypes']["$f_key"][] = array(
								'id'      => $data->data_id,
								'name'    => $data_formatted_name,
								'date'    => $data_formatted_date,
								'email'   => $data_formatted_email,
								'uploads' => $upload_array,
							);
							break;

						case 'feedback_large' :
						case 'feedback_small' :
							if ( ! isset( $exp_data['freetypes']["$f_key"] ) ) {
								$exp_data['freetypes']["$f_key"] = array();
							}

							if ( empty( $data->data->freetype[$f_key]['value'] ) ) {
								continue 2;
							}
							$exp_data['freetypes']["$f_key"][] = array(
								'value' => esc_textarea( $data->data->freetype["$f_key"]['value'] ),
								'name'  => $data_formatted_name,
								'email' => $data_formatted_email,
								'phone' => $data->data->phone,
								'date'  => $data_formatted_date,
								'id'    => $data->data_id,
							);
							break;
						case 'mathematical' :
							if ( ! isset( $exp_data['freetypes']["$f_key"] ) ) {
								$exp_data['freetypes']["$f_key"] = array();
							}
							if ( empty( $data->data->freetype[$f_key]['value'] ) ) {
								continue 2;
							}

							$exp_data['freetypes']["$f_key"][] = array(
								'value' => $data->freetype[$f_key]['settings']['prefix'] . ' ' . $data->data->freetype["$f_key"]['value'] . ' ' . $data->freetype[$f_key]['settings']['suffix'],
								'name'  => $data_formatted_name,
								'email' => $data_formatted_email,
								'phone' => $data->data->phone,
								'date'  => $data_formatted_date,
								'id'    => $data->data_id,
							);
							break;
						case 'gps' :
							if ( ! isset( $exp_data['freetypes']["$f_key"] ) ) {
								$exp_data['freetypes']["$f_key"] = array();
							}
							if ( ! is_numeric( $data->data->freetype[$f_key]['lat'] ) || ! is_numeric( $data->data->freetype[$f_key]['long'] ) ) {
								continue 2;
							}

							$exp_data['freetypes']["$f_key"][] = array(
								'name'          => $data_formatted_name,
								'email'         => $data_formatted_email,
								'date'          => $data_formatted_date,
								'id'            => $data->data_id,
								'map'           => sprintf( '//maps.googleapis.com/maps/api/staticmap?markers=%1$s,%2$s&zoom=%3$s&size=500x300&scale=2&key=%4$s', round( (float) $data->data->freetype[$f_key]['lat'], 6 ), round( (float) $data->data->freetype[$f_key]['long'], 6 ), $data->freetype[$f_key]['settings']['zoom'], $ipt_fsqm_settings['gplaces_api'] ),
								'lat'           => $data->data->freetype[$f_key]['lat'],
								'long'          => $data->data->freetype[$f_key]['long'],
								'location_name' => $data->data->freetype[$f_key]['location_name'],
								'phone'         => $data->data->phone,
							);
							break;
						case 'feedback_matrix' :
							if ( ! isset( $exp_data['freetypes']["$f_key"] ) ) {
								$exp_data['freetypes']["$f_key"] = array();
							}
							$break_data = true;
							$sanitized_matrix = array();
							foreach ( $data->data->freetype[$f_key]['rows'] as $r_key => $columns ) {
								$sanitized_matrix["$r_key"] = array();
								foreach ( (array) $columns as $c_key => $val ) {
									if ( $val != '' ) {
										$break_data = false;
									}
									$sanitized_matrix["$r_key"]["$c_key"] = esc_textarea( $val );
								}
							}
							if ( $break_data ) {
								continue 2;
							}
							$exp_data['freetypes']["$f_key"][] = array(
								'name'          => $data_formatted_name,
								'email'         => $data_formatted_email,
								'date'          => $data_formatted_date,
								'id'            => $data->data_id,
								'matrix'        => $sanitized_matrix,
								'phone'         => $data->data->phone,
							);
							break;
						case 'signature' :
							if ( ! isset( $exp_data['freetypes']["$f_key"] ) ) {
								$exp_data['freetypes']["$f_key"] = array();
							}
							if ( empty( $data->data->freetype[$f_key]['value'] ) || $data->data->freetype["$f_key"]['value'] == 'image/jsignature;base30,' ) {
								continue 2;
							}

							$exp_data['freetypes']["$f_key"][] = array(
								'value' => $data->convert_jsignature_image( $data->data->freetype["$f_key"]['value'], $data->freetype[ "$f_key" ]['settings']['color'] ),
								'name'  => $data_formatted_name,
								'email' => $data_formatted_email,
								'phone' => $data->data->phone,
								'date'  => $data_formatted_date,
								'id'    => $data->data_id,
							);
							break;
					}
				}
			}

			// Do the others
			if ( is_array( $pinfos ) && ! empty( $pinfos ) ) {
				foreach ( $pinfos as $p_key ) {
					$p_key = (int) $p_key;
					// Check if it exists
					if ( ! isset( $data->pinfo[ $p_key ] ) || ! isset( $data->data->pinfo[ $p_key ] ) ) {
						continue;
					}
					// Get the element
					$element = $data->pinfo[ $p_key ];

					// Avoid if it is conditionally hidden
					if ( isset( $data->conditional_hidden_blacklist[$element['m_type']] ) ) {
						if ( in_array( (string) $p_key, $data->conditional_hidden_blacklist[$element['m_type']] ) ) {
							continue;
						}
					}

					// Process accordingly
					switch ( $element['type'] ) {
						// Third Party Hooking
						default:
							$definition = $data->get_element_definition( $element );
							if ( isset( $definition['callback_report_calculator'] ) && is_callable( $definition['callback_report_calculator'] ) ) {
								if ( ! isset( $exp_data['pinfos'][ "$p_key" ] ) ) {
									$exp_data['pinfos'][ "$p_key" ] = array();
								}
								$exp_data['pinfos'][ "$p_key" ] = call_user_func( $definition['callback_report_calculator'], $element, $data->data->pinfo[ $p_key ], $p_key, $exp_data['pinfos'][ "$p_key" ] );
							}
							break;
						// Text types
						case 'f_name' :
						case 'l_name' :
						case 'email' :
						case 'phone' :
						case 'p_name' :
						case 'p_email' :
						case 'p_phone' :
						case 'textinput' :
						case 'textarea' :
						case 'password' :
						case 'keypad' :
						case 'hidden' :
							if ( ! isset( $exp_data['pinfos'][ "$p_key" ] ) ) {
								$exp_data['pinfos'][ "$p_key" ] = array();
							}

							if ( empty( $data->data->pinfo[$p_key]['value'] ) ) {
								continue 2;
							}

							$exp_data['pinfos'][ "$p_key" ][] = array(
								'value' => esc_textarea( $data->data->pinfo[ "$p_key" ]['value'] ),
								'name'  => $data_formatted_name,
								'email' => $data_formatted_email,
								'phone' => $data->data->phone,
								'date'  => $data_formatted_date,
								'id'    => $data->data_id,
							);
							break;
						// Datetime
						case 'datetime' :
							if ( ! isset( $exp_data['pinfos'][ "$p_key" ] ) ) {
								$exp_data['pinfos'][ "$p_key" ] = array();
							}
							if ( empty( $data->data->pinfo[$p_key]['value'] ) ) {
								continue 2;
							}

							$dtvalue = $data->data->pinfo[ "$p_key" ]['value'];
							$current_picker_timestamp = strtotime( $dtvalue );
							if ( false !== $current_picker_timestamp ) {
								switch ( $element['settings']['type'] ) {
									case 'date' :
										$dtvalue = date( $date_formats[$element['settings']['date_format']], $current_picker_timestamp );
										break;
									case 'time' :
										$dtvalue = date( $time_formats[$element['settings']['time_format']], $current_picker_timestamp );
										break;
									case 'datetime' :
										$dtvalue = date( $date_formats[$element['settings']['date_format']] . ' ' . $time_formats[$element['settings']['time_format']], $current_picker_timestamp );
										break;
								}
							}
							$exp_data['pinfos'][ "$p_key" ][] = array(
								'value' => $dtvalue,
								'name'  => $data_formatted_name,
								'email' => $data_formatted_email,
								'phone' => $data->data->phone,
								'date'  => $data_formatted_date,
								'id'    => $data->data_id,
							);
							break;
						// Address
						case 'address' :
							if ( ! isset( $exp_data['pinfos'][ "$p_key" ] ) ) {
								$exp_data['pinfos'][ "$p_key" ] = array();
							}
							if ( $data->data->pinfo[$p_key]['values'] == array(
								'recipient' => '',
								'line_one' => '',
								'line_two' => '',
								'line_three' => '',
								'country' => '',
							) ) {
								continue 2;
							}

							$address = $data->data->pinfo[$p_key]['values'];
							$address = array_map( 'esc_textarea', $address );
							$exp_data['pinfos'][ "$p_key" ][] = array(
								'values' => $address,
								'name'  => $data_formatted_name,
								'email' => $data_formatted_email,
								'phone' => $data->data->phone,
								'date'  => $data_formatted_date,
								'id'    => $data->data_id,
							);
							break;
						// Payment
						case 'payment' :
							// This needs special data collection
							// Get the data from table
							$payment_db = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$ipt_fsqm_info['payment_table']} WHERE data_id = %d", $data->data_id ) );
							if ( ! isset( $exp_data['pinfos'][ "$p_key" ] ) ) {
								$exp_data['pinfos'][ "$p_key" ] = array();
							}
							// Skip if not present
							if ( is_null( $payment_db ) ) {
								continue 2;
							}

							// Populate the data
							$invoiceid = str_replace( '{id}', $payment_db->id, $data->settings['payment']['invoicenumber'] );
							if ( $invoiceid == '' ) {
								$invoiceid = $payment_db->id;
							}
							$payment_status = IPT_FSQM_Form_Elements_Static::ipt_fsqm_get_payment_status();
							$payment_modes = IPT_FSQM_Form_Elements_Static::ipt_fsqm_get_payment_gateways();
							$exp_data['pinfos'][ "$p_key" ][] = array(
								'invoice' => $invoiceid,
								'status' => isset( $payment_status[ $payment_db->status ] ) ? $payment_status[ $payment_db->status ] : __( 'Unknown', 'ipt_fsqm' ),
								'txn' => ( is_null( $payment_db->txn ) ? __( 'N/A', 'ipt_fsqm' ) : $payment_db->txn ) ,
								'gateway' => isset( $payment_modes[ $payment_db->mode ] ) ? $payment_modes[ $payment_db->mode ] : __( 'Unknown', 'ipt_fsqm' ),
								'total' => $payment_db->amount,
								'name' => $data_formatted_name,
								'email' => $data_formatted_email,
								'phone' => $data->data->phone,
								'id' => $data->data_id,
								'date' => $data_formatted_date,
							);
							break;
						// MCQ
						case 'p_radio' :
						case 'p_checkbox' :
						case 'p_select' :
							if ( ! isset( $exp_data['pinfos'][ "$p_key" ] ) ) {
								$exp_data['pinfos'][ "$p_key" ] = array();
								$exp_data['pinfos'][ "$p_key" ]['others_data'] = array();
							}
							if ( empty( $data->data->pinfo[ $p_key ]['options'] ) ) {
								continue 2;
							}
							foreach ( $data->data->pinfo[ $p_key ]['options'] as $o_key ) {
								$exp_data['pinfos'][ "$p_key" ][ "$o_key" ] = isset( $exp_data['pinfos'][ "$p_key" ][ "$o_key" ] ) ? $exp_data['pinfos'][ "$p_key" ][ "$o_key" ] + 1 : 1;
							}
							if ( ! empty( $data->data->pinfo[ $p_key ]['others'] ) ) {
								$exp_data['pinfos'][ "$p_key" ]['others_data'][] = array(
									'value' => esc_textarea( $data->data->pinfo[ $p_key ]['others'] ),
									'name' => $data_formatted_name,
									'email' => $data_formatted_email,
									'id' => $data->data_id,
									'date' => $data_formatted_date,
								);
							}
							break;
						// Single State
						case 's_checkbox' :
							if ( ! isset( $exp_data['pinfos'][ "$p_key" ] ) ) {
								$exp_data['pinfos'][ "$p_key" ] = array(
									'checked' => 0,
									'unchecked' => 0,
								);
							}
							if ( ! isset( $data->data->pinfo[$p_key]['value'] ) ) {
								continue 2;
							}
							if ( $data->data->pinfo[$p_key]['value'] == false ) {
								$exp_data['pinfos'][ "$p_key" ]['unchecked']++;
							} else {
								$exp_data['pinfos'][ "$p_key" ]['checked']++;
							}
							break;
						// Sorting
						case 'p_sorting' :
							if ( ! isset( $exp_data['pinfos'][ "$p_key" ] ) ) {
								$exp_data['pinfos'][ "$p_key" ] = array(
									'preset' => 0,
									'other' => 0,
									'orders' => array(),
								);
							}
							if ( empty( $data->data->pinfo[ $p_key ]['order'] ) ) {
								continue 2;
							}
							$correct_order = implode( '-', array_keys( $data->pinfo[ $p_key ]['settings']['options'] ) );
							$user_order = implode( '-', $data->data->pinfo[ $p_key ]['order'] );
							if ( $correct_order == $user_order ) {
								$exp_data['pinfos'][ "$p_key" ]['preset']++;
							} else {
								$exp_data['pinfos'][ "$p_key" ]['other']++;
							}
							$exp_data['pinfos'][ "$p_key" ]['orders'][$user_order] = isset( $exp_data['pinfos'][ "$p_key" ]['orders'][$user_order] ) ? $exp_data['pinfos'][ "$p_key"]['orders'][$user_order] + 1 : 1;
							break;
						// Guest Blog
						case 'guestblog' :
							if ( ! isset( $exp_data['pinfo'][ "$p_key" ] ) ) {
								$exp_data['pinfo'][ "$p_key" ] = array();
							}

							if ( empty( $data->data->pinfo[ "$p_key" ]['value'] ) ) {
								continue 2;
							}

							$pinfo_text = array(
								'value' => $data->data->pinfo[ "$p_key" ]['value'],
								'title' => $data->data->pinfo[ "$p_key" ]['title'],
								'taxonomy' => array(),
								'email' => $data_formatted_email,
								'phone' => $data->data->phone,
								'id' => $data->data_id,
								'name' => $data_formatted_name,
								'date' => $data_formatted_date,
							);
							if ( ! empty( $data->data->pinfo[ "$p_key" ]['taxonomy'] ) ) {
								foreach ( $data->data->pinfo[ "$p_key" ]['taxonomy'] as $taxonomy => $tax_selected ) {
									$tax_data = get_taxonomy( $taxonomy );
									$terms_data = array();
									foreach( (array) $tax_selected as $term ) {
										$terms_data[] = get_term( $term, $taxonomy )->name;
									}
									$pinfo_text['taxonomy'][] = array(
										'tax' => $tax_data->labels->name,
										'terms' => $terms_data,
									);
								}
							}
							$exp_data['pinfos'][ "$p_key" ][] = $pinfo_text;
							break;
						case 'repeatable' :
							if ( ! isset( $exp_data['pinfo'][ "$p_key" ] ) ) {
								$exp_data['pinfo'][ "$p_key" ] = array();
							}

							if ( empty( $data->data->pinfo[ $p_key ]['values'] ) ) {
								continue 2;
							}

							$pinfo_text = array(
								'values' => array(),
								'name' => $data_formatted_name,
								'email' => $data_formatted_email,
								'phone' => $data->data->phone,
								'id' => $data->data_id,
								'date' => $data_formatted_date,
							);

							foreach ( (array) $data->data->pinfo[ $p_key ]['values'] as $i_key => $items ) {
								$new_value = array();
								foreach ( (array) $data->pinfo[ $p_key ]['settings']['group'] as $g_key => $group ) {
									if ( isset( $items[ $g_key ] ) ) {
										switch ( $group['type'] ) {
											case 'radio' :
												$new_value[ $g_key ] = str_replace( '__', ' ', $items[ $g_key ] );
												break;
											case 'checkbox' :
											case 'select' :
											case 'select_multiple' :
												$options = array();
												foreach ( (array) $items[ $g_key ] as $op ) {
													$options[] = str_replace( '__', ' ', $op );
												}
												$new_value[ $g_key ] = implode( '<br />', $options );
												break;
											case 'text' :
											case 'phone' :
											case 'url' :
											case 'email' :
											case 'number' :
											case 'integer' :
											case 'personName' :
											case 'password' :
											case 'textarea' :
												$new_value[ $g_key ] = $items[ $g_key ];
												break;
											case 'date' :
												$date = DateTime::createFromFormat( 'Y-m-d', $items[ $g_key ] );
												if ( ! $date ) {
													$new_value[ $g_key ] = $items[ $g_key ];
												} else {
													$new_value[ $g_key ] = $date->format( get_option( 'date_format' ) );
												}
												break;
											case 'time' :
												$date = DateTime::createFromFormat( 'H:i:s', $items[ $g_key ] );
												if ( ! $date ) {
													$new_value[ $g_key ] = $items[ $g_key ];
												} else {
													$new_value[ $g_key ] = $date->format( get_option( 'time_format' ) );
												}
												break;
											case 'datetime' :
												$date = DateTime::createFromFormat( 'Y-m-d H:i:s', $items[ $g_key ] );
												if ( ! $date ) {
													$new_value[ $g_key ] = $items[ $g_key ];
												} else {
													$new_value[ $g_key ] = $date->format( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ) );
												}
												break;
										}
									} else {
										$new_value[ $g_key ] = '';
									}
								}
								$pinfo_text['values'][ $i_key ] = $new_value;
							}

							$exp_data['pinfos'][ "$p_key" ][] = $pinfo_text;
							break;
					}
				}
			}
		}

		//Calculate the done
		$done_till_now = $doing * $per_page + $per_page;
		if ( $done_till_now >= $total ) {
			$return['done'] = 100;
		} else {
			$return['done'] = (float) $done_till_now * 100 / $total;
		}
		$return['debug_info'] = (object) $debug_info;

		// Update the database
		$wpdb->update( $ipt_fsqm_exp_info['exp_table'], array(
			'mcqs' => maybe_serialize( $exp_data['mcqs'] ),
			'freetypes' => maybe_serialize( $exp_data['freetypes'] ),
			'pinfos' => maybe_serialize( $exp_data['pinfos'] ),
			'complete' => $return['done'] == 100 ? 1 : 0,
		), array(
			'id' => $exp_id,
		), '%s', '%d' );

		echo json_encode( (object) $return );
		die();
	}

	/*==========================================================================
	 * Dashboard Widget Callback
	 *========================================================================*/

	/**
	 * Hooks to FSQM Pro Dashboard to show Exporter
	 *
	 * Works only with FSQM Pro version 2.1.1+
	 *
	 * @param IPT_FSQM_Admin $admin The instance of the admin class
	 * @return void
	 */
	public static function ipt_fsqm_dashboard( $admin ) {
		$admin->ui->iconbox( __( 'Exporter for eForm', 'ipt_fsqm_exp' ), array( __CLASS__, 'notify_about_available_features' ), 'download2' );
	}

	/*==========================================================================
	 * Managing PHP Resources & Export Types
	 *========================================================================*/

	public static function extend_resources($ignore_user_abort = false) {
		global $ipt_fsqm_exp_settings;
		$memory_limit = (int) $ipt_fsqm_exp_settings['memory'];
		$execution_time = (int) $ipt_fsqm_exp_settings['execution_time'];

		if ( $memory_limit < 128 ) {
			$memory_limit = 128;
		}

		if ( $execution_time < 100 ) {
			$execution_time = 100;
		}

		@ini_set( 'memory_limit', $memory_limit . 'M' );
		@ini_set( 'max_execution_time', $execution_time );
		@ignore_user_abort( $ignore_user_abort );
	}

	public static function notify_about_available_features() {
		global $ipt_fsqm_exp_info;
		$errors = array();
		$okays = array();

		IPT_FSQM_EXP_Export_API::extend_resources();

		//Check PHP Version
		if ( defined( 'PHP_VERSION' ) ) {
			$okays[] = sprintf( __( 'Current PHP version %s, required version 5.2.0.', 'ipt_fsqm_exp' ), PHP_VERSION );
		} else {
			$errors[] = __( 'Your server does not give PHP version information.', 'ipt_fsqm_exp' );
		}

		//Check LIBXML
		if ( defined( 'LIBXML_DOTTED_VERSION' ) ) {
			$okays[] = sprintf( __( 'XML Support is loaded and current LIBXML version is %s.', 'ipt_fsqm_exp' ), LIBXML_DOTTED_VERSION );
		} else {
			if ( defined( 'LIBXML_VERSION' ) ) {
				$okays[] = sprintf( __( 'XML Support is loaded and current LIBXML version is %s.', 'ipt_fsqm_exp' ), LIBXML_VERSION );
			} else {
				$okays[] = __( 'XML Support is loaded.', 'ipt_fsqm_exp' );
			}
		}

		//Check ZIP Extension
		if ( !extension_loaded( 'zip' ) ) {
			$errors[] = __( 'PHP ZIP extension is not loaded. Export Support Disabled.', 'ipt_fsqm_exp' );
		} else {
			$okays[] = __( 'PHP ZIP extension is loaded. Export support enabled.', 'ipt_fsqm_exp' );
		}

		//Check PHP GD
		if ( self::gd_version() < 2 ) {
			$errors[] = __( 'PHP GD2 is not loaded. XLS, PDF and HTML support disabled.', 'ipt_fsqm_exp' );
		} else {
			$okays[] = __( 'PHP GD2 is loaded. XLS, PDF and HTML support enabled.', 'ipt_fsqm_exp' );
		}

		//Check PHP Memory Limit
		if ( self::memory_limit() !== false ) {
			$memory_limit = self::memory_limit() / 1024 / 1024;
			if ( $memory_limit >= 1024 ) {
				$okays[] = sprintf( __( 'Server status: Excellent. PHP memory limit %dMB.', 'ipt_fsqm_exp' ), $memory_limit );
			} else if ( $memory_limit >= 512 ) {
					$okays[] = sprintf( __( 'Server status: Good. PHP memory limit %dMB.', 'ipt_fsqm_exp' ), $memory_limit );
				} else if ( $memory_limit >= 256 ) {
					$okays[] = sprintf( __( 'Server status: Fair. PHP memory limit %dMB.', 'ipt_fsqm_exp' ), $memory_limit );
				} else if ( $memory_limit >= 128 ) {
					$errors[] = sprintf( __( 'Server status: Average. PHP memory limit %dMB.', 'ipt_fsqm_exp' ), $memory_limit );
				} else {
				$errors[] = sprintf( __( 'Server status: Poor. PHP memory limit %dMB.', 'ipt_fsqm_exp' ), $memory_limit );
			}
		} else {
			$errors[] = __( 'Could not determine PHP memory limit.', 'ipt_fsqm_exp' );
		}

		//Check PHP Max Execution Time
		$max_execution = @ini_get( 'max_execution_time' );
		if ( $max_execution ) {
			if ( $max_execution >= 200 ) {
				$okays[] = sprintf( __( 'PHP execution time limit %dSeconds, which is excellent.', 'ipt_fsqm_exp' ), $max_execution );
			} else if ( $max_execution >= 100 ) {
					$okays[] = sprintf( __( 'PHP execution time limit %dSeconds, which is good.', 'ipt_fsqm_exp' ), $max_execution );
				} else if ( $max_execution >= 50 ) {
					$errors[] = sprintf( __( 'PHP execution time limit %dSeconds, may cause problem for PDFs.', 'ipt_fsqm_exp' ), $max_execution );
				} else {
				$errors[] = sprintf( __( 'PHP execution time limit %dSeconds, may cause problem for downloads.', 'ipt_fsqm_exp' ), $max_execution );
			}
		} else {
			$errors[] = __( 'Could not determine PHP execution time limit.', 'ipt_fsqm_exp' );
		}

		self::$ui = IPT_Plugin_UIF_Admin::instance( 'ipt_fsqm' );
		$buttons = array(
			'xlsx' => 'javascript:;',
			'pdf' => 'javascript:;',
			'xls' => 'javascript:;',
			'html' => 'javascript:;',
		);

		self::$ui->help_head( __( 'Plugin Version', 'ipt_fsqm_exp' ) );
		_e( 'If the Script version and DB version do not match, then deactivate the plugin and reactivate again. This should solve the problem. If the problem persists then contact the developer.', 'ipt_fsqm_exp' );
		self::$ui->help_tail();
?>
<p>
	<?php _e( 'Thank you for purchasing Exporter for eForm extension.', 'ipt_fsqm_exp' ); ?> |
	<?php printf( __( '<strong>Plugin Version:</strong> <em>%s(Script)/%s(DB)</em>', 'ipt_fsqm_exp' ), IPT_FSQM_EXP_Loader::$version, $ipt_fsqm_exp_info['version'] ); ?> |
	<strong><?php _e( 'Server Status', 'ipt_fsqm_exp' ); ?></strong> <span style="vertical-align: middle;" class="icon-arrow-down-2"></span>
</p>
		<?php
		self::$ui->clear();
		$i = 0;
		if ( !empty( $okays ) ) {
			foreach ( $okays as $okay ) {
				echo '<div class="' . ( $i % 2 == 0 ? 'ipt_uif_left_col' : 'ipt_uif_right_col' ) . '"><div class="ipt_uif_col_inner">';
				self::$ui->msg_okay( $okay );
				echo '</div></div>';
				$i++;
			}
		}
		if ( !empty( $errors ) ) {
			foreach ( $errors as $error ) {
				echo '<div class="' . ( $i % 2 == 0 ? 'ipt_uif_left_col' : 'ipt_uif_right_col' ) . '"><div class="ipt_uif_col_inner">';
				self::$ui->msg_error( $error );
				echo '</div></div>';
				$i++;
			}
		}
		self::$ui->clear();
?>
<div class="ipt_uif_float_left">
	<p><strong><?php _e( 'Available Export Formats', 'ipt_fsqm_exp' ); ?></strong></p>
</div>
<div class="ipt_uif_float_right">
	<?php echo self::render_available_export_formats_buttons( $buttons ); ?>
</div>
		<?php
		self::$ui->clear();
	}

	/**
	 * Get available export formats
	 * Depending on the installed libraries
	 *
	 *
	 * @return array list of all file extensions in terms of formats (xlsx, pdf, html, xls)
	 */
	public static function get_available_export_formats() {
		$formats = array();
		$formats[] = 'xlsx';
		if ( self::gd_version() >= 2 ) {
			$formats[] = 'pdf';
			$formats[] = 'html';
			$formats[] = 'xls';
		}
		return $formats;
	}

	/**
	 * Render available export format buttons
	 * Depending on the installed libraries
	 *
	 *
	 * @param array   $links Default anchor links to the corresponding buttons
	 * @param bool    $hidden Whether or not the button container is hidden
	 * @return string        The HTML of the buttons
	 */
	public static function render_available_export_formats_buttons( $links = array(), $hidden = false, $persistent_report = false ) {
		$formats = self::get_available_export_formats();
		$button_class = array(
			'xlsx' => 'file-openoffice',
			'pdf' => 'file-pdf',
			'xls' => 'file-excel',
			'html' => 'html5',
		);
		$button_text = array(
			'xlsx' => __( 'Open Office XML or Excel 2007 and above', 'ipt_fsqm_exp' ),
			'pdf' => __( 'Portable Document Format', 'ipt_fsqm_exp' ),
			'xls' => __( 'Excel 95 and above', 'ipt_fsqm_exp' ),
			'html' => __( 'HTML File', 'ipt_fsqm_exp' ),
		);
		ob_start();
?>
<div class="ipt_uif_button_container center"<?php echo ( $hidden ? ' style="display: none;"' : '' ); ?>>
	<?php if ( true == $persistent_report ) : ?>
		<a style="display: inline-block; float: none;" href="#" title="<?php _e( 'View the report in browser', 'ipt_fsqm_exp' ) ?>" class="ipt_uif_button secondary-button ipt_fsqm_exp_export_button_persistent_report"><span class="ipt-icomoon-stats"></span> <?php _e( 'View Report', 'ipt_fsqm' ); ?></a>
	<?php endif; ?>
	<?php foreach ( $button_class as $format => $class ) : ?>
	<?php if ( in_array( $format, $formats, true ) ) : ?>
	<a title="<?php echo esc_attr( $button_text[$format] ); ?>" style="display: inline-block; float: none;" href="<?php echo isset( $links[$format] ) ? $links[$format] : '#'; ?>" class="ipt_uif_button secondary-button ipt_fsqm_exp_export_button_<?php echo esc_attr( $format ); ?>"><span class="ipt-icomoon-<?php echo $class; ?>"></span> <?php echo strtoupper( $format ); ?></a>
	<?php endif; ?>
	<?php endforeach; ?>
</div>
		<?php
		return ob_get_clean();
	}

	/*==========================================================================
	 * Some Internal methods
	 *========================================================================*/

	public static function autop_array( &$item, $key ) {
		$item = wpautop( $item );
	}

	public static function delTree( $dir ) {
		$files = array_diff( scandir( $dir ), array( '.', '..' ) );
		foreach ( $files as $file ) {
			( is_dir( "$dir/$file" ) ) ? self::delTree( "$dir/$file" ) : @unlink( "$dir/$file" );
		}
		return @rmdir( $dir );
	}

	public static function memory_limit( $override = false ) {
		$memory_limit = @ini_get( 'memory_limit' );
		static $memory_byte = null;
		if ( !$memory_limit ) {
			return false;
		}

		if ( $memory_byte !== null && $override !== true ) {
			return $memory_byte;
		}

		if ( preg_match( '/^(\d+)(.)$/', $memory_limit, $matches ) ) {
			$memory_byte = $matches[1];
			switch ( $matches[2] ) {
			case 'G' :
				$memory_byte = $memory_byte * 1024 * 1024 * 1024;
				break;
			case 'M':
				$memory_byte = $memory_byte * 1024 * 1024;
				break;
			case 'K':
				$memory_byte = $memory_byte * 1024;
				break;
			default :
				//According to @link http://www.php.net/manual/en/faq.using.php#faq.using.shorthandbytes
				//anything else will mean it is set in bytes
				//So we do not have to do anything
			}
		} else {
			if ( is_numeric( $memory_limit ) ) {
				$memory_byte = $memory_limit;
			} else {
				//Revert to default
				if ( version_compare( '5.2.0', PHP_VERSION, '>' ) ) {
					$memory_byte = 128 * 1024 * 1024;
				} else {
					$memory_byte = 16 * 1024 * 1024;
				}
			}
		}
		return $memory_byte;
	}

	/**
	 * Get version of GD
	 *
	 * @link http://www.php.net/manual/en/function.gd-info.php#52481
	 *
	 * @param integer $user_ver Specify version, do this only if specification is 1
	 * @return integer            Version number of GD library (0 if not installed, otherwise 1 or 2)
	 */
	public static function gd_version( $user_ver = 0 ) {
		if ( ! extension_loaded( 'gd' ) ) { return; }
		static $gd_ver = 0;
		// Just accept the specified setting if it's 1.
		if ( $user_ver == 1 ) { $gd_ver = 1; return 1; }
		// Use the static variable if function was called previously.
		if ( $user_ver !=2 && $gd_ver > 0 ) { return $gd_ver; }
		// Use the gd_info() function if possible.
		if ( function_exists( 'gd_info' ) ) {
			$ver_info = gd_info();
			preg_match( '/\d/', $ver_info['GD Version'], $match );
			$gd_ver = $match[0];
			return $match[0];
		}
		// If phpinfo() is disabled use a specified / fail-safe choice...
		if ( preg_match( '/phpinfo/', ini_get( 'disable_functions' ) ) ) {
			if ( $user_ver == 2 ) {
				$gd_ver = 2;
				return 2;
			} else {
				$gd_ver = 1;
				return 1;
			}
		}
		// ...otherwise use phpinfo().
		ob_start();
		phpinfo( 8 );
		$info = ob_get_contents();
		ob_end_clean();
		$info = stristr( $info, 'gd version' );
		preg_match( '/\d/', $info, $match );
		$gd_ver = $match[0];
		if ( $gd_ver === null ) {
			$gd_ver = 0;
		}
		return $gd_ver;
	}
}
